// Copyright CloudQuery Authors
// This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0.
// If a copy of the MPL was not distributed with this file, You can obtain one at https://mozilla.org/MPL/2.0/.

package client

import (
	"fmt"

	"github.com/aliyun/alibaba-cloud-sdk-go/services/bssopenapi"
	"github.com/aliyun/alibaba-cloud-sdk-go/services/cms"
	"github.com/aliyun/alibaba-cloud-sdk-go/services/ecs"
	"github.com/aliyun/aliyun-oss-go-sdk/oss"
	"github.com/cloudquery/cloudquery/plugins/source/alicloud/client/services"
	sdkretryablehttp "github.com/cloudquery/plugin-sdk/v4/helpers/retryablehttp"
	"github.com/hashicorp/go-retryablehttp"
	"github.com/rs/zerolog"
)

type Services struct {
	OSS services.OssClient
	BSS services.BssopenapiClient
	ECS services.EcsClient
	CMS services.CmsClient
}

func initServices(logger zerolog.Logger, account AccountSpec, region string) (*Services, error) {
	retryClient := retryablehttp.NewClient()
	retryClient.Logger = sdkretryablehttp.NewLeveledLogger(logger)
	client := retryClient.StandardClient()

	ossCli, err := oss.New(fmt.Sprintf("oss-%s.aliyuncs.com", region), account.AccessKey, account.SecretKey, oss.Timeout(15, 30), oss.HTTPClient(client))
	if err != nil {
		return nil, fmt.Errorf("failed to initialize oss client: %w", err)
	}
	// At the time of writing this comment, RetryTimes doesn't seem to be implemented in the OSS client
	// Therefore we use go-retryablehttp to handle retries, and set the retry times to 0 in case it gets implemented in the future
	ossCli.Config.RetryTimes = 0

	bssCli, err := bssopenapi.NewClientWithAccessKey(region, account.AccessKey, account.SecretKey)
	if err != nil {
		return nil, fmt.Errorf("failed to initialize bssopenapi client: %w", err)
	}
	// The BSS client implements the retry logic, so we enable it
	bssCli.GetConfig().WithAutoRetry(true)

	ecsCli, err := ecs.NewClientWithAccessKey(region, account.AccessKey, account.SecretKey)
	if err != nil {
		return nil, fmt.Errorf("failed to initialize ecs client: %w", err)
	}
	// The ECS client implements the retry logic, so we enable it
	ecsCli.GetConfig().WithAutoRetry(true)

	cmsCli, err := cms.NewClientWithAccessKey(region, account.AccessKey, account.SecretKey)
	if err != nil {
		return nil, fmt.Errorf("failed to initialize cms client: %w", err)
	}
	// The CMS client implements the retry logic, so we enable it
	cmsCli.GetConfig().WithAutoRetry(true)

	return &Services{
		OSS: ossCli,
		BSS: bssCli,
		ECS: ecsCli,
		CMS: cmsCli,
	}, nil
}
