// Copyright CloudQuery Authors
// This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0.
// If a copy of the MPL was not distributed with this file, You can obtain one at https://mozilla.org/MPL/2.0/.

package client

import (
	_ "embed"
	"errors"
	"fmt"

	"github.com/cloudquery/plugin-sdk/v4/scheduler"
)

type Spec struct {
	Accounts          []AccountSpec `json:"accounts,omitempty" jsonschema:"required,minItems=1"`
	BillHistoryMonths int           `json:"bill_history_months,omitempty"`
	// A best effort maximum number of Go routines to use. Lower this number to reduce memory usage.
	Concurrency int `json:"concurrency,omitempty"`
	// The scheduler to use when determining the priority of resources to sync. By default, it is set to `dfs`.
	//
	// For more information about this, see [performance tuning](/docs/advanced-topics/performance-tuning).
	Scheduler *scheduler.Strategy `json:"scheduler,omitempty" jsonschema:"default=dfs"`
}

type AccountSpec struct {
	Name      string   `json:"name,omitempty" jsonschema:"required,minLength=1"`
	Regions   []string `json:"regions,omitempty" jsonschema:"required,minItems=1"`
	AccessKey string   `json:"access_key,omitempty" jsonschema:"required,minLength=1"`
	SecretKey string   `json:"secret_key,omitempty" jsonschema:"required,minLength=1"`
}

func (s *Spec) SetDefaults() {
	if s.Concurrency == 0 {
		s.Concurrency = 50000
	}
	if s.BillHistoryMonths == 0 {
		s.BillHistoryMonths = 12
	}
	if s.Scheduler == nil {
		defaultStrategy := scheduler.StrategyDFS
		s.Scheduler = &defaultStrategy
	}
}

func (s *Spec) Validate() error {
	if len(s.Accounts) == 0 {
		return errors.New("missing alicloud accounts in configuration")
	}
	accountNames := map[string]struct{}{}
	for _, account := range s.Accounts {
		if account.Name == "" {
			return errors.New("missing alicloud account name in configuration")
		}
		if _, found := accountNames[account.Name]; found {
			return fmt.Errorf("duplicate alicloud account name %s in configuration", account.Name)
		}
		accountNames[account.Name] = struct{}{}
		if account.AccessKey == "" {
			return fmt.Errorf("missing access_key in account configuration for account %s", account.Name)
		}
		if account.SecretKey == "" {
			return fmt.Errorf("missing secret_key in account configuration for account %s", account.Name)
		}
		if len(account.Regions) == 0 {
			return fmt.Errorf("missing regions in account configuration for account %s", account.Name)
		}
	}
	return nil
}

//go:embed schema.json
var JSONSchema string
