> **Copyright © CloudQuery Authors**  
> **License:** This Source Code Form is subject to the terms of the [Mozilla Public License, v. 2.0](https://mozilla.org/MPL/2.0/).

# AWS Source Plugin Contribution Guide

Thanks for contributing to CloudQuery! You are awesome. This document serves as a guide for adding new services and resources to the AWS source plugin.

There are two main steps to adding a new AWS resource:

1. [Generate interfaces for the AWS SDK function(s) that fetch the resource](#step-1-generate-interfaces-for-the-aws-sdk-functions-that-fetch-the-resource)
2. [Add a new table](#step-2-add-a-new-table)

As a prerequisite, in [aws-sdk-go-v2](https://pkg.go.dev/github.com/aws/aws-sdk-go-v2) ensure API calls exist to list/describe the desired resource, and make note of:

- to which aws service the resource belongs
- the schema of the returned object(s)

## Step 1. Generate Interfaces for the AWS SDK Function(s) that Fetch the Resource

### Generate the Service Interface (if it doesn't exist)

1. Check in [client/services.go](client/services.go) that the service you need has an interface defined. If it does, you can skip to [Step 2](#step-2-add-a-new-table). If not, read on to learn how to generate the interface.
2. Inside [codegen/main.go](codegen/main.go), add the client for the AWS SDK you need to the `clients` slice. You may need to run `go get github.com/aws/aws-sdk-go-v2/service/<service-name>` (e.g. `go get github.com/aws/aws-sdk-go-v2/service/dynamodb`) to add the dependency first.
3. Run `make gen-mocks`. This takes a few seconds, but it will create a mock for it that will be used in unit tests later
4. Add the new service to end of the list of values for `AllAWSServiceNames` in [client/service_names.go](client/service_names.go). (This will also require you to create a new constant AWSService<ServiceName>)
5. Add a case for the new service name you just added to `InitService` and `GetService` [client/services.go](client/services.go)


### Step 2. Add a New Table

The process to follow for adding a new table is:

1. Add a new directory matching the AWS service name under [resources/services](resources/services) (e.g. `resources/services/newservice`), if one doesn't exist already
2. Ensure permissions are added to `tools/generate_iam_permissions` and not directly to `table_permissions.json` (which is generated by `generate_iam_permissions`).
3. Create a new file under the new directory with the name of the resource (e.g. `resources/services/newservice/myresource.go`) and add a function that returns `*schema.Table`. The easiest is to copy-paste an existing table as a starting point ([`Kinesis`](resources/services/kinesis/kinesis.go) is a good example).
4. **Important**: Add a call to the new function to the list of tables in [tables.go](resources/plugin/tables.go). Otherwise, the new table will not be included in the plugin.  
5. Update all the fields, taking special care to ensure that the `transformers.TransformWithStruct()` call in the `Resolver` function has the correct struct type (e.g. `transformers.TransformWithStruct(&types.MyResource{})`)
6. Implement the resolver function. This should have the signature: 
   ```go
   func fetchMyResource(ctx context.Context, meta schema.ClientMeta, parent *schema.Resource, res chan<- interface{}) error {
       // TODO: implement this
   }
   ```
   
   The easiest is to copy-paste an existing resolver as a starting point. (Again, [`Kinesis`](resources/services/kinesis/streams_fetch.go) is a good example.)
   
   You may use a type assertion on `meta` to obtain a reference to your interface functions, e.g.:
   ```go
   svc := meta.(*client.Client).Services().MyService
   ```
   
   With this in hand, complete the resolver function to fetch all resources. After resources are retrieved, send them to the `res` channel for the SDK to deliver to all destinations.
6. Implement a mock test in `myresource_mock_test.go`. We will not describe this in detail here; look at a few examples for similar resources to get you started.

We highly recommend looking at other resources similar to yours to get an idea of what needs to be done in each step.  

### Implementing Resolver Functions

A few important things to note when adding functions that call the AWS API:

- If possible, always use an API call that allows you to fetch many resources at once
- Take pagination into account. Use `Paginator`s if the AWS service supports it. Ensure you fetch **all** the resources.
- Columns may also have their own resolver functions (not covered in this guide). This may be used for simple transformations or when additional calls can help add further context to the table.
- Many resources require a `List` call, followed by a `Describe` call. Look for examples using `PreResourceResolver` to see the canonical way of implementing this. (In short: the table resolver function will call `List`, while the `PreResourceResolver`, called once per resource, will call `Describe`)


### Adding Support for a new streaming event:

1. Add Support for filtering the API Requests at the table level by ensuring that `table_options` supports the desired table.
  - Add new option in `client/tableoptions/templates/main.ResolverDefinitions`
  - ensure `goimports` is installed
  - Run `make gen-table-options`
2. Write new test for table option that you just added
3. Ensure all tests in the `table_options` package pass. (You might need to add a new value to skip to the `TestTableOptionsUnmarshal` test)
4. Update the Table resolver to use the new table option. Typically, this will follow the following pattern:
    ```go
    svc := cl.Services(client.AWSService<SERVICE>).<SERVICE>
	for _, w := range cl.Spec.TableOptions.<RESOURCE>.Filters() {
		paginator := <SERVICE>.New<API-CALL>Paginator(svc, &w.<API-CALL>Input)
		for paginator.HasMorePages() {
			page, err := paginator.NextPage(ctx, func(options *<SERVICE>.Options) {
				options.Region = cl.Region
			})
			if err != nil {
				return err
			}
			res <- page.<RESOURCE>
		}
	}
	return nil
    ```

5. Find an event that you are adding. Easiest way is to look in the EventHistory section of the Cloudtrail Console... This contains all events that occurred in an AWS account for the last 90 days
6. Create the following files under the service that generates the event:
    - `resources/streaming/<SERVICE>/<EVENT_NAME>.go`
    - `resources/streaming/<SERVICE>/<EVENT_NAME>_test.go`
7. The `<EVENT_NAME>.go` file should implement three things:
    - a `struct` that can hold the event data
    - a `function` named `EventKeys` that returns a `[]string` of the keys that are used to identify if a specific event is supported for streaming sync
    - a `function` named `SyncingOptions` that returns all of the information required to run a sync and return the minimal amount of data required
8. The `<EVENT_NAME>_test.go` file should implement a test that ensures that the event is supported for streaming sync
    Be sure to change any accountIDs to just `testAccount` and the region must be `us-east-1`
9. Add the new event to the output of `supportedEvents` in `resources/plugin/client.go`


## General Tips

- Keep transformations to a minimum. As far as possible, we aim to deliver an accurate reflection of what the AWS API provides.
- We generally only unroll structs one level deep. Nested structs should be transformed into JSON columns. 
- For consistency, make sure the resource has an `ARN` stored in a column named `arn`. Sometimes this means using the AWS SDK to generate an ARN for the resource.
- Make sure the resource has a `tags` JSON column (if possible). Sometimes this requires additional SDK calls. Sometimes the column needs to be renamed from `tag_list` to `tags` (and converted to a map). There are custom `ResolveTags` and `ResolveTagFields` resolvers to help with this. It's not always possible, but we try to keep the `tags` column consistent across AWS resources.
- Before submitting a pull request, run `make gen-docs` to generate documentation for the table. Include these generated files in the pull request.
- If you get stuck or need help, feel free to reach out on Slack.
- Ensure you add a table multiplexer when adding the new table (unless the new table is a relation of a parent table, which has its own multiplexer).
