// Copyright CloudQuery Authors
// This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0.
// If a copy of the MPL was not distributed with this file, You can obtain one at https://mozilla.org/MPL/2.0/.

package tableoptions

import (
	"reflect"
	"slices"
)

type changeCaseFunc func(string) string

// skipFields is a list of fields that should not be changed. This is useful for fields that are
// maps, where case needs to be preserved. Skipfields are passed to nested objects so all matches no matter the level are skipped.
func changeCaseForObject(obj any, changeCase changeCaseFunc, skipFields ...string) {
	effectiveSkip := make([]string, len(skipFields))
	for i, v := range skipFields {
		effectiveSkip[i] = changeCase(v)
	}
	value := reflect.ValueOf(obj)
	switch value.Kind() {
	case reflect.Map:
		iter := value.MapRange()
		for iter.Next() {
			k := iter.Key()
			if k.Kind() == reflect.String {
				nk := changeCase(k.String())
				v := iter.Value()
				if slices.Contains(effectiveSkip, nk) {
					continue
				}
				changeCaseForObject(v.Interface(), changeCase, skipFields...)
				value.SetMapIndex(k, reflect.Value{})
				value.SetMapIndex(reflect.ValueOf(nk), v)
			}
		}
	case reflect.Slice:
		for i := 0; i < value.Len(); i++ {
			changeCaseForObject(value.Index(i).Interface(), changeCase, skipFields...)
		}
	}
}
