// Copyright CloudQuery Authors
// This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0.
// If a copy of the MPL was not distributed with this file, You can obtain one at https://mozilla.org/MPL/2.0/.

package plugin

import (
	"encoding/json"
	"fmt"
	"maps"
	"reflect"
	"regexp"
	"slices"
	"strings"

	"github.com/cloudquery/cloudquery/plugins/source/aws/client/spec"
	"github.com/cloudquery/cloudquery/plugins/source/aws/client/spec/tableoptions"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/accessanalyzer"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/account"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/acm"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/acmpca"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/amp"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/amplify"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/apigateway"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/apigatewayv2"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/appconfig"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/appflow"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/applicationautoscaling"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/appmesh"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/apprunner"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/appstream"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/appsync"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/athena"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/auditmanager"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/autoscaling"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/autoscalingplans"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/backup"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/backupgateway"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/batch"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/bedrock"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/bedrockagent"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/budgets"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/cloudformation"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/cloudfront"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/cloudhsmv2"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/cloudtrail"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/cloudwatch"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/cloudwatchlogs"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/codeartifact"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/codebuild"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/codecommit"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/codedeploy"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/codegurureviewer"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/codepipeline"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/cognito"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/comprehend"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/computeoptimizer"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/config"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/connect"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/costexplorer"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/datapipeline"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/datasync"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/dax"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/detective"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/devopsguru"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/directconnect"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/directoryservice"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/dlm"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/dms"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/docdb"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/dynamodb"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/dynamodbstreams"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/ec2"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/ecr"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/ecrpublic"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/ecs"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/efs"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/eks"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/elasticache"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/elasticbeanstalk"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/elasticsearch"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/elastictranscoder"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/elbv1"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/elbv2"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/emr"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/eventbridge"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/firehose"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/fis"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/frauddetector"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/freetier"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/fsx"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/glacier"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/globalaccelerator"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/glue"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/grafana"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/guardduty"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/health"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/healthlake"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/iam"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/identitystore"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/imagebuilder"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/inspector"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/inspector2"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/invoicing"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/iot"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/kafka"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/kendra"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/keyspaces"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/kinesis"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/kinesisanalytics"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/kms"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/lakeformation"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/lambda"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/lexmodels"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/lexmodelsv2"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/lightsail"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/location"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/macie2"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/memorydb"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/mq"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/mwaa"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/neptune"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/networkfirewall"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/networkmanager"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/opensearch"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/organizations"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/pinpoint"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/polly"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/qldb"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/quicksight"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/ram"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/rds"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/redshift"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/rekognition"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/resiliencehub"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/resourcegroups"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/route53"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/route53recoverycontrolconfig"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/route53recoveryreadiness"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/route53resolver"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/s3"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/s3tables"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/s3vectors"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/sagemaker"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/savingsplans"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/scheduler"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/secretsmanager"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/securityhub"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/servicecatalog"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/servicediscovery"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/servicequotas"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/ses"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/shield"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/signer"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/snowball"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/sns"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/sqs"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/ssm"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/ssmincidents"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/ssoadmin"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/stepfunctions"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/storagegateway"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/support"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/swf"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/timestream"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/transcribe"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/transfer"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/trustedadvisor"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/vpclattice"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/waf"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/wafregional"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/wafv2"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/wellarchitected"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/workspaces"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/xray"
	"github.com/cloudquery/plugin-sdk/v4/caser"
	"github.com/cloudquery/plugin-sdk/v4/docs"
	"github.com/cloudquery/plugin-sdk/v4/premium"
	"github.com/cloudquery/plugin-sdk/v4/schema"
	"github.com/cloudquery/plugin-sdk/v4/transformers"
	"github.com/cloudquery/plugin-sdk/v4/types"
	orderedmap "github.com/wk8/go-ordered-map/v2"

	schemaDocs "github.com/cloudquery/codegen/jsonschema/docs"
	invoschema "github.com/invopop/jsonschema"
)

var awsExceptions = map[string]string{
	"accessanalyzer":               "AWS Identity and Access Management (IAM) Access Analyzer",
	"acm":                          "Amazon Certificate Manager (ACM)",
	"acmpca":                       "AWS Certificate Manager Private Certificate Authority (ACM PCA)",
	"amp":                          "Amazon Managed Service for Prometheus (AMP)",
	"apigateway":                   "Amazon API Gateway",
	"apigatewayv2":                 "Amazon API Gateway v2",
	"appconfig":                    "AWS AppConfig",
	"appflow":                      "Amazon AppFlow",
	"applicationautoscaling":       "Application Auto Scaling",
	"appmesh":                      "AWS App Mesh",
	"apprunner":                    "AWS App Runner",
	"appstream":                    "Amazon AppStream",
	"arn":                          "Amazon Resource Name (ARN)",
	"arns":                         "Amazon Resource Names (ARNs)",
	"auditmanager":                 "AWS Audit Manager",
	"autoscaling":                  "Auto Scaling",
	"autoscalingplans":             "Auto Scaling Plans",
	"aws":                          "", // remove "AWS" from names, because in most cases it will be replaced with either Amazon or AWS
	"backupgateway":                "AWS Backup gateway",
	"byoasns":                      "Autonomous System Number (ASN) and bring your own IP addresses (BYOIP) CIDR associations",
	"byoip":                        "Bring your own IP addresses (BYOIP)",
	"cloudformation":               "AWS CloudFormation",
	"cloudhsm":                     "AWS CloudHSM",
	"cloudhsmv2":                   "AWS CloudHSM v2",
	"cloudtrail":                   "AWS CloudTrail",
	"cloudwatchlogs":               "AWS CloudWatch Logs",
	"codeartifact":                 "AWS CodeArtifact",
	"codebuild":                    "AWS CodeBuild",
	"codecommit":                   "AWS CodeCommit",
	"codedeploy":                   "AWS CodeDeploy",
	"computeoptimizer":             "Compute Optimizer",
	"costexplorer":                 "AWS Cost Explorer",
	"datasync":                     "AWS DataSync",
	"datapipeline":                 "AWS Data Pipeline",
	"detective":                    "Amazon Detective",
	"directconnect":                "AWS Direct Connect",
	"directoryservice":             "AWS Directory Service",
	"dlm":                          "AWS Data Lifecycle Manager",
	"docdb":                        "Amazon DocumentDB",
	"dynamodb":                     "Amazon DynamoDB",
	"dynamodbstreams":              "Amazon DynamoDB",
	"ebs":                          "Amazon Elastic Block Store (EBS)",
	"ec2":                          "Amazon Elastic Compute Cloud (EC2)",
	"ecr":                          "Amazon Elastic Container Registry (ECR)",
	"ecs":                          "Amazon Elastic Container Service (ECS)",
	"efs":                          "Amazon Elastic File System (EFS)",
	"eks":                          "Amazon Elastic Kubernetes Service (EKS)",
	"elasticbeanstalk":             "AWS Elastic Beanstalk",
	"elastictranscoder":            "Amazon Elastic Transcoder",
	"elb":                          "Amazon Elastic Load Balancer (ELB)",
	"elbv1":                        "Amazon Elastic Load Balancer (ELB) v1",
	"elbv2":                        "Amazon Elastic Load Balancer (ELB) v2",
	"emr":                          "Amazon EMR",
	"eventbridge":                  "Amazon EventBridge",
	"frauddetector":                "Amazon Fraud Detector",
	"fis":                          "Amazon Fault Injection Service (FIS)",
	"fsx":                          "Amazon FSx",
	"guardduty":                    "Amazon GuardDuty",
	"globalaccelerator":            "AWS Global Accelerator",
	"identitystore":                "Identity Store",
	"imagebuilder":                 "EC2 Image Builder",
	"ipam":                         "VPC IP Address Manager (IPAM)",
	"ipams":                        "VPC IP Address Managers (IPAMs)",
	"iot":                          "AWS IoT",
	"kms":                          "AWS Key Management Service (AWS KMS)",
	"lambda":                       "AWS Lambda",
	"lexv2":                        "Amazon Lex V2",
	"location":                     "Amazon Location Service",
	"mq":                           "Amazon MQ",
	"mwaa":                         "Amazon MWAA",
	"nat":                          "NAT",
	"networkmanager":               "Network Manager",
	"pinpoint":                     "Amazon Pinpoint",
	"qldb":                         "Quantum Ledger Database (QLDB)",
	"quicksight":                   "QuickSight",
	"rds":                          "Amazon Relational Database Service (RDS)",
	"rekognition":                  "Amazon Rekognition",
	"resiliencehub":                "AWS Resilience Hub",
	"route53":                      "Amazon Route 53",
	"route53recoverycontrolconfig": "Amazon Route 53 Application Recovery Controller Recovery Control Configuration",
	"route53recoveryreadiness":     "Amazon Route 53 Application Recovery Controller Recovery Readiness",
	"route53resolver":              "Amazon Route 53 Resolver",
	"sagemaker":                    "Amazon SageMaker",
	"secretsmanager":               "AWS Secrets Manager",
	"securityhub":                  "AWS Security Hub",
	"servicecatalog":               "AWS Service Catalog",
	"servicediscovery":             "AWS Cloud Map",
	"ses":                          "Amazon Simple Email Service (SES)",
	"s3vectors":                    "Amazon S3 Vectors",
	"s3tables":                     "Amazon S3 Tables",
	"signer":                       "AWS Signer",
	"ssm":                          "AWS Systems Manager (SSM)",
	"ssmincidents":                 "AWS Systems Manager (SSM) Incident Manager",
	"swf":                          "Amazon Simple Workflow Service (SWF)",
	"transfer":                     "AWS Transfer Family",
	"vpc-lattice":                  "VPC Lattice",
	"wellarchitected":              "AWS Well-Architected",
	"xray":                         "AWS X-Ray",
}

func titleTransformer() func(*schema.Table) {
	exceptions := maps.Clone(docs.DefaultTitleExceptions)
	maps.Copy(exceptions, awsExceptions)
	csr := caser.New(caser.WithCustomExceptions(exceptions))
	return func(table *schema.Table) {
		titleTransformerFunc(table, csr)
	}
}

func titleTransformerFunc(table *schema.Table, csr *caser.Caser) {
	if len(table.Title) == 0 {
		table.Title = strings.Trim(strings.ReplaceAll(csr.ToTitle(table.Name), "  ", " "), " ")
	}
	for _, rel := range table.Relations {
		titleTransformerFunc(rel, csr)
	}
}

func descriptionTransformer() func(*schema.Table) {
	var sc invoschema.Schema
	if err := json.Unmarshal([]byte(spec.JSONSchema), &sc); err != nil {
		return nil
	}

	tableNamesToOptionsDocs := make(map[string]string)
	tableOptionsType := reflect.ValueOf(&tableoptions.TableOptions{}).Elem().Type()

	tocRegex := regexp.MustCompile(`# Table of contents[\s\S]+?##`)

	for i := range tableOptionsType.NumField() {
		field := tableOptionsType.Field(i)
		fieldType := strings.Split(field.Type.String(), ".")[1]
		defValue, ok := sc.Definitions[fieldType]
		if !ok {
			panic("definition not found for " + field.Name)
		}
		tableName := strings.Split(field.Tag.Get("json"), ",")[0]
		if tableName == "" {
			panic("table name not found for " + field.Name)
		}

		// generating schema can override the original schema
		// so we need to remove the skip comment to ensure everything gets printed as expected
		for _, def := range sc.Definitions {
			removeSkipComment(reflect.ValueOf(def))
		}
		newRoot := sc
		newRoot.ID = "Table Options"
		newRoot.Ref = "#/$defs/" + "Table Options"
		newRoot.Definitions["Table Options"] = defValue

		doc, _ := schemaDocs.GenerateFromSchema(newRoot, 1)

		tableNamesToOptionsDocs[tableName] = tocRegex.ReplaceAllString(doc, "##")
	}

	return func(table *schema.Table) {
		if tableNamesToOptionsDocs[table.Name] != "" {
			table.Description = table.Description + "\n\n" + tableNamesToOptionsDocs[table.Name]
		}
	}
}

func getTables(s *spec.Spec) schema.Tables {
	t := schema.Tables{
		accessanalyzer.Analyzers(),
		account.AlternateContacts(),
		account.Contacts(),
		acm.Certificates(),
		acmpca.CertificateAuthorities(),
		amp.Workspaces(),
		amplify.Apps(),
		apigateway.ApiKeys(),
		apigateway.ClientCertificates(),
		apigateway.DomainNames(),
		apigateway.RestApis(),
		apigateway.UsagePlans(),
		apigateway.VpcLinks(),
		apigatewayv2.Apis(),
		apigatewayv2.DomainNames(),
		apigatewayv2.VpcLinks(),
		appconfig.Applications(),
		appconfig.DeploymentStrategies(),
		appflow.Flows(),
		applicationautoscaling.Policies(),
		applicationautoscaling.ScalableTargets(),
		applicationautoscaling.ScalingActivities(),
		applicationautoscaling.ScheduledActions(),
		appmesh.Meshes(),
		apprunner.AutoScalingConfigurations(),
		apprunner.Connections(),
		apprunner.ObservabilityConfigurations(),
		apprunner.Services(),
		apprunner.VpcConnectors(),
		apprunner.VpcIngressConnections(),
		appstream.AppBlocks(),
		appstream.Applications(),
		appstream.DirectoryConfigs(),
		appstream.Fleets(),
		appstream.ImageBuilders(),
		appstream.Images(),
		appstream.Stacks(),
		appstream.UsageReportSubscriptions(),
		appstream.Users(),
		appsync.GraphqlApis(),
		athena.DataCatalogs(),
		athena.WorkGroups(),
		auditmanager.Assessments(),
		autoscaling.Groups(),
		autoscaling.LaunchConfigurations(),
		autoscaling.ScheduledActions(),
		autoscalingplans.Plans(),
		backup.GlobalSettings(),
		backup.Jobs(),
		backup.Plans(),
		backup.ProtectedResources(),
		backup.RegionSettings(),
		backup.ReportPlans(),
		backup.Vaults(),
		backupgateway.Gateways(),
		batch.ComputeEnvironments(),
		batch.JobDefinitions(),
		batch.JobQueues(),
		bedrockagent.Agents(),
		bedrock.CustomModels(),
		bedrock.EvaluationJobs(),
		bedrock.FoundationModels(),
		bedrock.Guardrails(),
		bedrock.InferenceProfiles(),
		bedrock.ModelCopyJobs(),
		bedrock.ModelCustomizationJobs(),
		bedrock.ProvisionedModelThroughputs(),
		budgets.Actions(),
		budgets.Budgets(),
		cloudformation.Stacks(),
		cloudformation.StackSets(),
		cloudfront.CachePolicies(),
		cloudfront.Distributions(),
		cloudfront.Functions(),
		cloudfront.KeyValueStores(),
		cloudfront.OriginAccessIdentities(),
		cloudfront.OriginRequestPolicies(),
		cloudfront.ResponseHeaderPolicies(),
		cloudhsmv2.Backups(),
		cloudhsmv2.Clusters(),
		cloudtrail.Channels(),
		cloudtrail.Events(),
		cloudtrail.Imports(),
		cloudtrail.Trails(),
		cloudwatch.Alarms(),
		cloudwatch.Metrics(),
		cloudwatch.MetricData(),
		cloudwatch.MetricStreams(),
		cloudwatchlogs.Deliveries(),
		cloudwatchlogs.DeliveryDestinations(),
		cloudwatchlogs.DeliverySources(),
		cloudwatchlogs.LogGroups(),
		cloudwatchlogs.MetricFilters(),
		cloudwatchlogs.ResourcePolicies(),
		codeartifact.Domains(),
		codeartifact.Repositories(),
		codebuild.Projects(),
		codebuild.SourceCredentials(),
		codecommit.Repositories(),
		codedeploy.Applications(),
		codedeploy.DeploymentConfigs(),
		codedeploy.Deployments(),
		codegurureviewer.RepositoryAssociations(),
		codepipeline.Pipelines(),
		codepipeline.Webhooks(),
		cognito.IdentityPools(),
		cognito.UserPools(),
		comprehend.DocumentClassificationJobs(),
		comprehend.DocumentClassifiers(),
		comprehend.DominantLanguageDetectionJobs(),
		comprehend.Endpoints(),
		comprehend.EntitiesDetectionJobs(),
		comprehend.EntityRecognizers(),
		comprehend.EventsDetectionJobs(),
		comprehend.Flywheels(),
		comprehend.KeyPhrasesDetectionJobs(),
		comprehend.PiiEntitiesDetectionJobs(),
		comprehend.SentimentDetectionJobs(),
		comprehend.TargetedSentimentDetectionJobs(),
		comprehend.TopicsDetectionJobs(),
		computeoptimizer.AutoscalingGroupsRecommendations(),
		computeoptimizer.EbsVolumeRecommendations(),
		computeoptimizer.Ec2InstanceRecommendations(),
		computeoptimizer.EcsServiceRecommendations(),
		computeoptimizer.EnrollmentStatuses(),
		computeoptimizer.LambdaFunctionsRecommendations(),
		connect.Instances(),
		config.ConfigRules(),
		config.ConfigurationAggregators(),
		config.ConfigurationRecorders(),
		config.ConformancePacks(),
		config.DeliveryChannels(),
		config.RetentionConfigurations(),
		costexplorer.CustomCost(),
		costexplorer.ThirtyDayCost(),
		costexplorer.ThirtyDayCostForecast(),
		datapipeline.Pipelines(),
		datasync.Agents(),
		datasync.Locations(),
		dax.Clusters(),
		detective.Graphs(),
		devopsguru.Insights(),
		devopsguru.MonitoredResources(),
		devopsguru.NotificationChannels(),
		directconnect.Connections(),
		directconnect.Gateways(),
		directconnect.Lags(),
		directconnect.Locations(),
		directconnect.VirtualGateways(),
		directconnect.VirtualInterfaces(),
		directoryservice.Directories(),
		dms.Certificates(),
		dms.EventSubscriptions(),
		dms.ReplicationInstances(),
		dms.ReplicationSubnetGroups(),
		dms.ReplicationTasks(),
		docdb.Certificates(),
		docdb.ClusterParameterGroups(),
		docdb.Clusters(),
		docdb.EngineVersions(),
		docdb.EventCategories(),
		docdb.Events(),
		docdb.EventSubscriptions(),
		docdb.GlobalClusters(),
		docdb.PendingMaintenanceActions(),
		docdb.SubnetGroups(),
		dynamodb.Backups(),
		dynamodb.Exports(),
		dynamodb.GlobalTables(),
		dynamodb.Tables(),
		dlm.DlmLifecyclePolicies(),
		dynamodbstreams.Streams(),
		ec2.AccountAttributes(),
		ec2.AvailabilityZones(),
		ec2.ByoipCidrs(),
		ec2.CapacityReservations(),
		ec2.CustomerGateways(),
		ec2.DHCPOptions(),
		ec2.EbsDefaultKmsKeyId(),
		ec2.EbsEncryptionByDefault(),
		ec2.EbsSnapshots(),
		ec2.EbsVolumes(),
		ec2.EbsVolumesStatuses(),
		ec2.EgressOnlyInternetGateways(),
		ec2.Eips(),
		ec2.FlowLogs(),
		ec2.Hosts(),
		ec2.ImageBlockPublicAccessStates(),
		ec2.Images(),
		ec2.InstanceConnectEndpoints(),
		ec2.InstanceCreditSpecifications(),
		ec2.Instances(),
		ec2.InstanceStatuses(),
		ec2.InstanceTypes(),
		ec2.InternetGateways(),
		ec2.IPAMBYOASNs(),
		ec2.IPAMPools(),
		ec2.IPAMResourceDiscoveries(),
		ec2.IPAMResourceDiscoveryAssociations(),
		ec2.IPAMs(),
		ec2.IPAMScopes(),
		ec2.KeyPairs(),
		ec2.LaunchTemplates(),
		ec2.ManagedPrefixLists(),
		ec2.NatGateways(),
		ec2.NetworkAcls(),
		ec2.NetworkInterfaces(),
		ec2.PrefixLists(),
		ec2.RegionalConfigs(),
		ec2.Regions(),
		ec2.ReplaceRootVolumeTasks(),
		ec2.ReservedInstances(),
		ec2.RouteTables(),
		ec2.SecurityGroups(),
		ec2.SerialConsoleAccessStatus(),
		ec2.SnapshotBlockPublicAccessStates(),
		ec2.SpotFleetRequests(),
		ec2.SpotInstanceRequests(),
		ec2.Subnets(),
		ec2.TrafficMirrorFilters(),
		ec2.TrafficMirrorSessions(),
		ec2.TrafficMirrorTargets(),
		ec2.TransitGateways(),
		ec2.TransitGatewayConnectPeers(),
		ec2.VpcEndpointConnections(),
		ec2.VpcEndpoints(),
		ec2.VpcEndpointServiceConfigurations(),
		ec2.VpcEndpointServices(),
		ec2.VpcPeeringConnections(),
		ec2.Vpcs(),
		ec2.VpnConnections(),
		ec2.VpnGateways(),
		ecr.PullThroughCacheRules(),
		ecr.Registries(),
		ecr.RegistryPolicies(),
		ecr.Repositories(),
		ecrpublic.Repositories(),
		ecs.Clusters(),
		ecs.TaskDefinitions(),
		efs.AccessPoints(),
		efs.Filesystems(),
		eks.Clusters(),
		eks.AccessPolicies(),
		elasticache.Clusters(),
		elasticache.EngineVersions(),
		elasticache.Events(),
		elasticache.GlobalReplicationGroups(),
		elasticache.ParameterGroups(),
		elasticache.ReplicationGroups(),
		elasticache.ReservedCacheNodes(),
		elasticache.ReservedCacheNodesOfferings(),
		elasticache.ServerlessCaches(),
		elasticache.ServerlessCacheSnapshots(),
		elasticache.ServiceUpdates(),
		elasticache.Snapshots(),
		elasticache.SubnetGroups(),
		elasticache.UpdateActions(),
		elasticache.UserGroups(),
		elasticache.Users(),
		elasticbeanstalk.Applications(),
		elasticbeanstalk.ApplicationVersions(),
		elasticbeanstalk.Environments(),
		elasticbeanstalk.PlatformVersions(),
		elasticsearch.Domains(),
		elasticsearch.Packages(),
		elasticsearch.ReservedInstances(),
		elasticsearch.Versions(),
		elasticsearch.VpcEndpoints(),
		elastictranscoder.Pipelines(),
		elastictranscoder.Presets(),
		elbv1.LoadBalancers(),
		elbv2.LoadBalancers(),
		elbv2.TargetGroups(),
		emr.BlockPublicAccessConfigs(),
		emr.Clusters(),
		emr.ReleaseLabels(),
		emr.SecurityConfigurations(),
		emr.Studios(),
		eventbridge.ApiDestinations(),
		eventbridge.Archives(),
		eventbridge.Connections(),
		eventbridge.Endpoints(),
		eventbridge.EventBuses(),
		eventbridge.EventSources(),
		eventbridge.Replays(),
		firehose.DeliveryStreams(),
		fis.Actions(),
		fis.Experiments(),
		fis.ExperimentTemplates(),
		fis.TargetResourceTypes(),
		frauddetector.BatchImports(),
		frauddetector.BatchPredictions(),
		frauddetector.Detectors(),
		frauddetector.EntityTypes(),
		frauddetector.EventTypes(),
		frauddetector.ExternalModels(),
		frauddetector.Labels(),
		frauddetector.Models(),
		frauddetector.Outcomes(),
		frauddetector.Variables(),
		freetier.Usage(),
		fsx.Backups(),
		fsx.DataRepositoryAssociations(),
		fsx.DataRepositoryTasks(),
		fsx.FileCaches(),
		fsx.FileSystems(),
		fsx.Snapshots(),
		fsx.StorageVirtualMachines(),
		fsx.Volumes(),
		glacier.DataRetrievalPolicies(),
		glacier.Vaults(),
		globalaccelerator.Accelerators(),
		globalaccelerator.CustomRoutingAccelerators(),
		glue.Classifiers(),
		glue.Connections(),
		glue.Crawlers(),
		glue.Databases(),
		glue.DatacatalogEncryptionSettings(),
		glue.DevEndpoints(),
		glue.Jobs(),
		glue.MlTransforms(),
		glue.Registries(),
		glue.SecurityConfigurations(),
		glue.Triggers(),
		glue.Workflows(),
		grafana.Workspaces(),
		guardduty.Detectors(),
		health.HealthEvents(health.LookupRegionViaDNS),
		health.HealthOrganizationEvents(health.LookupRegionViaDNS),
		healthlake.FHIRDataStores(),
		iam.AccountAuthorizationDetails(),
		iam.Accounts(),
		iam.CredentialReports(),
		iam.Groups(),
		iam.InstanceProfiles(),
		iam.OpenidConnectIdentityProviders(),
		iam.PasswordPolicies(),
		iam.Policies(),
		iam.Roles(),
		iam.SamlIdentityProviders(),
		iam.ServerCertificates(),
		iam.Users(),
		iam.VirtualMfaDevices(),
		identitystore.Groups(),
		identitystore.Users(),
		imagebuilder.DistributionConfigurations(),
		imagebuilder.Images(),
		imagebuilder.Workflows(),
		inspector.Findings(),
		inspector2.CisScans(),
		inspector2.CoveredResources(),
		inspector2.Findings(),
		invoicing.InvoiceUnits(),
		iot.BillingGroups(),
		iot.CaCertificates(),
		iot.Certificates(),
		iot.Jobs(),
		iot.Policies(),
		iot.SecurityProfiles(),
		iot.Streams(),
		iot.ThingGroups(),
		iot.Things(),
		iot.ThingTypes(),
		iot.TopicRules(),
		kafka.Clusters(),
		kafka.Configurations(),
		kendra.Indices(),
		keyspaces.Keyspaces(),
		kinesis.Streams(),
		kinesisanalytics.Applications(),
		kms.Aliases(),
		kms.Keys(),
		lakeformation.DataCellsFilters(),
		lakeformation.OptIns(),
		lakeformation.Permissions(),
		lakeformation.Resources(),
		lakeformation.Transactions(),
		lambda.Functions(),
		lambda.Layers(),
		lambda.Runtimes(),
		lexmodels.Bots(),
		lexmodels.BuiltinIntents(),
		lexmodels.BuiltinSlotTypes(),
		lexmodels.Intents(),
		lexmodels.Migrations(),
		lexmodels.SlotTypes(),
		lexmodelsv2.Bots(),
		lightsail.Alarms(),
		lightsail.Buckets(),
		lightsail.Certificates(),
		lightsail.ContainerServices(),
		lightsail.Databases(),
		lightsail.DatabaseSnapshots(),
		lightsail.Disks(),
		lightsail.Distributions(),
		lightsail.Instances(),
		lightsail.InstanceSnapshots(),
		lightsail.LoadBalancers(),
		lightsail.StaticIps(),
		location.Keys(),
		location.Maps(),
		macie2.AllowLists(),
		macie2.AutomatedDiscoveryAccounts(),
		macie2.ClassificationJobs(),
		macie2.ClassificationScopes(),
		macie2.CustomDataIdentifiers(),
		macie2.Findings(),
		macie2.Invitations(),
		macie2.ManagedDataIdentifiers(),
		macie2.Members(),
		macie2.SensitivityInspectionTemplates(),
		macie2.UsageTotals(),
		memorydb.ReservedNodes(),
		mq.Brokers(),
		mwaa.Environments(),
		neptune.ClusterParameterGroups(),
		neptune.Clusters(),
		neptune.ClusterSnapshots(),
		neptune.DbParameterGroups(),
		neptune.EventSubscriptions(),
		neptune.GlobalClusters(),
		neptune.Instances(),
		neptune.SubnetGroups(),
		networkfirewall.FirewallPolicies(),
		networkfirewall.Firewalls(),
		networkfirewall.RuleGroups(),
		networkfirewall.TLSInspectionConfigurations(),
		networkmanager.Attachments(),
		networkmanager.CoreNetworks(),
		networkmanager.GlobalNetworks(),
		opensearch.Domains(),
		opensearch.InboundConnections(),
		opensearch.OutboundConnections(),
		opensearch.ReservedInstances(),
		opensearch.Versions(),
		opensearch.VPCEndpoints(),
		organizations.Accounts(),
		organizations.DelegatedAdministrators(),
		organizations.OrganizationalUnits(),
		organizations.Organizations(),
		organizations.Policies(),
		organizations.ResourcePolicies(),
		organizations.Roots(),
		pinpoint.Apps(),
		qldb.Ledgers(),
		quicksight.Analyses(),
		quicksight.Dashboards(),
		quicksight.DataSets(),
		quicksight.DataSources(),
		quicksight.Folders(),
		quicksight.Groups(),
		quicksight.Templates(),
		quicksight.Users(),
		polly.Lexicons(),
		polly.Voices(),
		polly.SpeechSynthesisTasks(),
		ram.Principals(),
		ram.Resources(),
		ram.ResourceShareAssociations(),
		ram.ResourceShareInvitations(),
		ram.ResourceShares(),
		ram.ResourceTypes(),
		rds.Certificates(),
		rds.ClusterParameterGroups(),
		rds.Clusters(),
		rds.ClusterSnapshots(),
		rds.DBParameterGroups(),
		rds.DBProxies(),
		rds.DBProxyEndpoints(),
		rds.DBSecurityGroups(),
		rds.DBSnapshots(),
		rds.EngineVersions(),
		rds.Events(),
		rds.EventSubscriptions(),
		rds.GlobalClusters(),
		rds.Instances(),
		rds.OptionGroups(),
		rds.PendingMaintenanceActions(),
		rds.ReservedInstances(),
		rds.SubnetGroups(),
		rds.MajorEngineVersions(),
		redshift.Clusters(),
		redshift.DataShares(),
		redshift.Events(),
		redshift.EventSubscriptions(),
		redshift.ReservedNodes(),
		redshift.SubnetGroups(),
		rekognition.Collections(),
		rekognition.MediaAnalysisJobs(),
		rekognition.Projects(),
		rekognition.StreamProcessors(),
		resiliencehub.Apps(),
		resiliencehub.ResiliencyPolicies(),
		resiliencehub.SuggestedResiliencyPolicies(),
		resourcegroups.ResourceGroups(),
		route53.DelegationSets(),
		route53.Domains(),
		route53.HealthChecks(),
		route53.HostedZones(),
		route53.Operations(),
		route53.TrafficPolicies(),
		route53recoverycontrolconfig.Clusters(),
		route53recoverycontrolconfig.ControlPanels(),
		route53recoveryreadiness.Cells(),
		route53recoveryreadiness.ReadinessChecks(),
		route53recoveryreadiness.RecoveryGroups(),
		route53recoveryreadiness.ResourceSets(),
		route53resolver.FirewallConfigs(),
		route53resolver.FirewallDomainLists(),
		route53resolver.FirewallRuleGroupAssociations(),
		route53resolver.FirewallRuleGroups(),
		route53resolver.ResolverEndpoints(),
		route53resolver.ResolverQueryLogConfigAssociations(),
		route53resolver.ResolverQueryLogConfigs(),
		route53resolver.ResolverRuleAssociations(),
		route53resolver.ResolverRules(),
		s3.AccessGrantInstances(),
		s3.AccessPoints(),
		s3.Accounts(),
		s3.Buckets(),
		s3.DirectoryBuckets(),
		s3.MultiRegionAccessPoints(),
		s3.StorageLensConfigurations(),
		s3.StorageLensGroups(),
		s3vectors.Buckets(),
		s3tables.Buckets(),
		sagemaker.Apps(),
		sagemaker.Domains(),
		sagemaker.EndpointConfigurations(),
		sagemaker.Endpoints(),
		sagemaker.HyperParameterTuningJobs(),
		sagemaker.Models(),
		sagemaker.NotebookInstances(),
		sagemaker.NotebookInstanceLifecycleConfigs(),
		sagemaker.StudioLifecycleConfigs(),
		sagemaker.TrainingJobs(),
		savingsplans.Plans(),
		scheduler.ScheduleGroups(),
		scheduler.Schedules(),
		secretsmanager.Secrets(),
		securityhub.EnabledStandards(),
		securityhub.Findings(),
		securityhub.Hubs(),
		servicecatalog.Portfolios(),
		servicecatalog.Products(),
		servicecatalog.ProvisionedProducts(),
		servicediscovery.Namespaces(),
		servicediscovery.Services(),
		servicequotas.Services(),
		ses.ActiveReceiptRuleSets(),
		ses.ConfigurationSets(),
		ses.ContactLists(),
		ses.CustomVerificationEmailTemplates(),
		ses.Identities(),
		ses.Templates(),
		ses.SuppressedDestinations(),
		shield.Attacks(),
		shield.ProtectionGroups(),
		shield.Protections(),
		shield.Subscriptions(),
		signer.Profiles(),
		snowball.Addresses(),
		snowball.Clusters(),
		snowball.CompatibleImages(),
		snowball.Jobs(),
		snowball.LongTermPricing(),
		snowball.PickupLocations(),
		sns.Subscriptions(),
		sns.Topics(),
		sqs.Queues(),
		ssm.Associations(),
		ssm.CommandInvocations(),
		ssm.ComplianceSummaryItems(),
		ssm.Documents(),
		ssm.Instances(),
		ssm.Inventories(),
		ssm.InventoryEntries(),
		ssm.InventorySchemas(),
		ssm.Parameters(),
		ssm.PatchBaselines(),
		ssm.Sessions(),
		ssmincidents.Incidents(),
		ssmincidents.ResponsePlans(),
		ssoadmin.Instances(),
		stepfunctions.Activities(),
		stepfunctions.StateMachines(),
		storagegateway.Gateways(),
		storagegateway.CacheReports(),
		storagegateway.Tapes(),
		storagegateway.TapePools(),
		support.Cases(),
		support.Services(),
		support.SeverityLevels(),
		support.TrustedAdvisorChecks(),
		swf.Domains(),
		timestream.Databases(),
		transcribe.CallAnalyticsCategories(),
		transcribe.CallAnalyticsJobs(),
		transcribe.LanguageModels(),
		transcribe.MedicalScribeJobs(),
		transcribe.MedicalTranscriptionJobs(),
		transcribe.MedicalVocabularies(),
		transcribe.TranscriptionJobs(),
		transcribe.Vocabularies(),
		transcribe.VocabularyFilters(),
		transfer.Certificates(),
		transfer.Connectors(),
		transfer.Profiles(),
		transfer.Servers(),
		transfer.Workflows(),
		trustedadvisor.Recommendations(),
		vpclattice.ResourceGateways(),
		vpclattice.Services(),
		vpclattice.ServiceNetworks(),
		waf.IPSets(),
		waf.RuleGroups(),
		waf.Rules(),
		waf.SubscribedRuleGroups(),
		waf.WebAcls(),
		wafregional.RateBasedRules(),
		wafregional.RuleGroups(),
		wafregional.Rules(),
		wafregional.WebAcls(),
		wafv2.Ipsets(),
		wafv2.ManagedRuleGroups(),
		wafv2.RegexPatternSets(),
		wafv2.RuleGroups(),
		wafv2.WebAcls(),
		wellarchitected.Lenses(),
		wellarchitected.ShareInvitations(),
		wellarchitected.Workloads(),
		workspaces.ConnectionAliases(),
		workspaces.Directories(),
		workspaces.Workspaces(),
		xray.EncryptionConfigs(),
		xray.Groups(),
		xray.ResourcePolicies(),
		xray.SamplingRules(),
	}

	if err := transformers.TransformTables(t); err != nil {
		panic(err)
	}
	transformTitles := titleTransformer()

	transformDescriptions := descriptionTransformer()

	for _, table := range t {
		schema.AddCqIDs(table)
		if s != nil && s.AddCQClientID {
			schema.AddCqClientID(table)
		}
		transformTitles(table)
		transformDescriptions(table)
		if err := validateTagsIsJSON(table); err != nil {
			panic(err)
		}
	}

	markPaid(t, s)
	return t
}

func validateTagsIsJSON(table *schema.Table) error {
	for _, col := range table.Columns {
		if col.Name == "tags" && col.Type != types.ExtensionTypes.JSON {
			return fmt.Errorf("column %s in table %s must be of type %s", col.Name, table.Name, types.ExtensionTypes.JSON)
		}
	}
	for _, rel := range table.Relations {
		if err := validateTagsIsJSON(rel); err != nil {
			return err
		}
	}

	return nil
}

func markPaid(tables schema.Tables, s *spec.Spec) {
	premium.MakeAllTablesPaid(tables)
	makeFree(tables)

	if s == nil || s.TableOptions == nil {
		return
	}

	set := s.TableOptions.TablesSet()
	if len(set) == 0 {
		return
	}

	toUpdate := make(schema.Tables, 0, len(set))
	for _, name := range set {
		table := tables.Get(name)
		if table != nil {
			toUpdate = append(toUpdate, table)
		}
	}

	premium.MakeAllTablesPaid(toUpdate) // updates in-place
}

var freeTables = []string{
	"aws_emr_supported_instance_types",
	"aws_servicequotas_quotas",
	"aws_rds_cluster_parameter_group_parameters",
	"aws_rds_cluster_parameters",
	"aws_ec2_instance_types",
	"aws_elasticache_reserved_cache_nodes_offerings",
	"aws_rds_engine_versions",
	"aws_iam_policy_versions",
	"aws_servicequotas_services",
	"aws_emr_release_labels",
	"aws_iam_policy_default_versions",
	"aws_ram_resource_types",
	"aws_appstream_images",
	"aws_elasticache_service_updates",
	"aws_elastictranscoder_presets",
	"aws_elasticsearch_versions",
	"aws_elasticsearch_packages",
	"aws_ssm_patch_baselines",
	"aws_rds_cluster_parameter_groups",
	"aws_ec2_managed_prefix_lists",
	"aws_availability_zones",
	"aws_regions",
}

func makeFree(tables schema.Tables) {
	for _, t := range tables {
		if slices.Contains(freeTables, t.Name) {
			t.IsPaid = false
		}
		makeFree(t.Relations)
	}
}

// removeSkipComment recursively searches for a Comment fields set to `skip_description` and then removes it
func removeSkipComment(v reflect.Value) {
	// Handle nil pointers
	if !v.IsValid() {
		return
	}

	// Dereference pointer if needed
	if v.Kind() == reflect.Ptr {
		if v.IsNil() {
			return
		}
		removeSkipComment(v.Elem())
		return
	}

	switch v.Kind() {
	case reflect.Struct:
		t := v.Type()
		// Recursively examine each field in the struct
		for i := range t.NumField() {
			field := t.Field(i)
			fieldName := field.Name
			// Skip certain fields
			if !slices.Contains([]string{"Then", "Else", "DependentSchemas", "PrefixItems", "Items", "Contains", "Properties", "PatternProperties", "AdditionalProperties", "PropertyNames", "ContentSchema", "Definitions", "AllOf", "AnyOf", "OneOf", "Not", "If", "Comments"}, fieldName) {
				continue
			}
			fieldValue := v.Field(i)

			switch fieldName {
			case "Comments":
				if fieldValue.IsValid() && fieldValue.Kind() == reflect.String && fieldValue.String() == "skip_description" {
					fieldValue.SetString("")
				}
			case "Properties":
				props := fieldValue.Interface().(*orderedmap.OrderedMap[string, *invoschema.Schema])
				for prop := props.Oldest(); prop != nil; prop = prop.Next() {
					removeSkipComment(reflect.ValueOf(prop.Value))
				}
			default:
				removeSkipComment(fieldValue)
			}
		}
	case reflect.Slice, reflect.Array:
		for i := range v.Len() {
			removeSkipComment(v.Index(i))
		}
	}
}
