// Copyright CloudQuery Authors
// This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0.
// If a copy of the MPL was not distributed with this file, You can obtain one at https://mozilla.org/MPL/2.0/.

package iam

import (
	"context"
	"strconv"

	"github.com/aws/aws-sdk-go-v2/aws/arn"

	"github.com/apache/arrow-go/v18/arrow"
	"github.com/aws/aws-sdk-go-v2/aws"
	"github.com/aws/aws-sdk-go-v2/service/iam"
	"github.com/aws/aws-sdk-go-v2/service/iam/types"
	"github.com/cloudquery/cloudquery/plugins/source/aws/client"
	"github.com/cloudquery/cloudquery/plugins/source/aws/client/spec/tableoptions"
	"github.com/cloudquery/plugin-sdk/v4/schema"
	"github.com/cloudquery/plugin-sdk/v4/transformers"
	sdkTypes "github.com/cloudquery/plugin-sdk/v4/types"
	"github.com/mitchellh/hashstructure/v2"
)

type iamPolicyWrapper struct {
	types.Policy
	InputJSON tableoptions.CustomIAMListPoliciesInput `json:"input_json"`
	InputHash string                                  `json:"input_hash"`
}

func Policies() *schema.Table {
	tableName := "aws_iam_policies"
	return &schema.Table{
		Name:              tableName,
		PermissionsNeeded: client.TablePermissions(tableName),
		Description:       `https://docs.aws.amazon.com/IAM/latest/APIReference/API_ManagedPolicyDetail.html`,
		Resolver:          fetchIamPolicies,
		Transform:         transformers.TransformWithStruct(&iamPolicyWrapper{}, transformers.WithUnwrapAllEmbeddedStructs(), transformers.WithPrimaryKeyComponents("InputHash", "Arn")),
		Multiplex:         client.ServiceAccountRegionMultiplexer(tableName, "iam"),
		Columns: []schema.Column{
			client.DefaultAccountIDColumn(true),
			{
				Name:                "id",
				Type:                arrow.BinaryTypes.String,
				Resolver:            schema.PathResolver("PolicyId"),
				PrimaryKeyComponent: true,
			},
			{
				Name:     "tags",
				Type:     sdkTypes.ExtensionTypes.JSON,
				Resolver: resolveIamPolicyTags,
			},
		},
		Relations: []*schema.Table{
			policyLastAccessedDetails(),
			policyVersions(),
			policyDefaultVersions(),
		},
	}
}

func fetchIamPolicies(ctx context.Context, meta schema.ClientMeta, parent *schema.Resource, res chan<- any) error {
	cl := meta.(*client.Client)
	svc := cl.Services(client.AWSServiceIam).Iam

	// the ListPolicies API doesn't allow for effective filtering, so we bypass that call and go straight to the get
	if len(cl.Spec.TableOptions.IAMPolicies.GetPolicyFilters()) > 0 {
		for _, filter := range cl.Spec.TableOptions.IAMPolicies.GetPolicyFilters() {
			policy, err := svc.GetPolicy(ctx, &filter.GetPolicyInput, func(options *iam.Options) {
				options.Region = cl.Region
			})
			if err != nil {
				return err
			}
			filterHash, err := hashstructure.Hash(&filter, hashstructure.FormatV2, nil)
			if err != nil {
				return err
			}
			res <- iamPolicyWrapper{
				Policy:    *policy.Policy,
				InputHash: strconv.FormatUint(filterHash, 10),
			}
		}
		return nil
	}

	for _, filter := range cl.Spec.TableOptions.IAMPolicies.Filters() {
		filterHash, err := hashstructure.Hash(&filter, hashstructure.FormatV2, nil)
		if err != nil {
			return err
		}

		paginator := iam.NewListPoliciesPaginator(svc, &filter.ListPoliciesInput)
		for paginator.HasMorePages() {
			page, err := paginator.NextPage(ctx, func(options *iam.Options) {
				options.Region = cl.Region
			})
			if err != nil {
				return err
			}

			policies := make([]iamPolicyWrapper, len(page.Policies))
			for idx, policy := range page.Policies {
				policies[idx] = iamPolicyWrapper{
					Policy:    policy,
					InputJSON: filter,
					InputHash: strconv.FormatUint(filterHash, 10),
				}
			}
			res <- policies
		}
	}
	return nil
}

func resolveIamPolicyTags(ctx context.Context, meta schema.ClientMeta, resource *schema.Resource, c schema.Column) error {
	r := resource.Item.(iamPolicyWrapper)
	cl := meta.(*client.Client)
	svc := cl.Services(client.AWSServiceIam).Iam

	parsedArn, err := arn.Parse(aws.ToString(r.Arn))
	if err != nil {
		return err
	}
	// AWS Managed Policies do not include an accountID in the ARN and cannot have any tags so no need to call API
	if parsedArn.AccountID != cl.AccountID {
		return nil
	}
	response, err := svc.ListPolicyTags(ctx, &iam.ListPolicyTagsInput{PolicyArn: r.Arn}, func(options *iam.Options) {
		options.Region = cl.Region
	})
	if err != nil {
		if cl.IsNotFoundError(err) {
			return nil
		}
		return err
	}
	return resource.Set("tags", client.TagsToMap(response.Tags))
}
