// Copyright CloudQuery Authors
// This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0.
// If a copy of the MPL was not distributed with this file, You can obtain one at https://mozilla.org/MPL/2.0/.

package route53

import (
	"context"
	"fmt"
	"strings"

	"github.com/apache/arrow-go/v18/arrow"
	"github.com/aws/aws-sdk-go-v2/aws"
	"github.com/aws/aws-sdk-go-v2/aws/arn"
	"github.com/aws/aws-sdk-go-v2/service/route53"
	"github.com/aws/aws-sdk-go-v2/service/route53/types"
	"github.com/cloudquery/cloudquery/plugins/source/aws/client"
	"github.com/cloudquery/cloudquery/plugins/source/aws/client/spec/tableoptions"
	"github.com/cloudquery/plugin-sdk/v4/schema"
	"github.com/cloudquery/plugin-sdk/v4/transformers"
)

const (
	maxNumberOfTagResourceIds = 10
)

func HostedZones() *schema.Table {
	tableName := "aws_route53_hosted_zones"
	return &schema.Table{
		Name:              tableName,
		PermissionsNeeded: client.TablePermissions(tableName),
		Description:       `https://docs.aws.amazon.com/Route53/latest/APIReference/API_HostedZone.html`,
		Resolver:          fetchRoute53HostedZones,
		Transform: transformers.TransformWithStruct(
			&route53HostedZoneWrapper{},
			transformers.WithUnwrapStructFields("HostedZone"),
			transformers.WithNameTransformer(client.CreateReplaceTransformer(map[string]string{"vp_cs": "vpcs"})),
		),
		Multiplex: client.ServiceAccountRegionMultiplexer(tableName, "route53"),
		Columns: []schema.Column{
			client.DefaultAccountIDColumn(false),
			{
				Name:                "arn",
				Type:                arrow.BinaryTypes.String,
				Resolver:            resolveRoute53HostedZoneArn,
				PrimaryKeyComponent: true,
			},
		},

		Relations: []*schema.Table{
			hostedZoneQueryLoggingConfigs(),
			hostedZoneResourceRecordSets(),
			hostedZoneTrafficPolicyInstances(),
		},
	}
}

type route53HostedZoneWrapper struct {
	types.HostedZone
	Tags            map[string]string
	DelegationSetId *string
	DelegationSet   *types.DelegationSet
	VPCs            []types.VPC
}

func fetchRoute53HostedZones(ctx context.Context, meta schema.ClientMeta, _ *schema.Resource, res chan<- any) error {
	cl := meta.(*client.Client)
	svc := cl.Services(client.AWSServiceRoute53).Route53
	opts := cl.Spec.TableOptions.Route53HostedZones

	processHostedZonesBundle := func(hostedZones []tableoptions.Route53ListHostedZonesInput) error {
		tagsCfg := &route53.ListTagsForResourcesInput{ResourceType: types.TagResourceTypeHostedzone, ResourceIds: make([]string, 0, len(hostedZones))}
		for i := range hostedZones {
			parsedId := strings.Replace(*hostedZones[i].Id, fmt.Sprintf("/%s/", types.TagResourceTypeHostedzone), "", 1)
			hostedZones[i].Id = &parsedId
			tagsCfg.ResourceIds = append(tagsCfg.ResourceIds, parsedId)
		}

		tagsResponse, err := svc.ListTagsForResources(ctx, tagsCfg, func(options *route53.Options) {
			options.Region = cl.Region
		})
		if err != nil {
			return err
		}

		for _, h := range hostedZones {
			gotHostedZone, err := svc.GetHostedZone(ctx, &route53.GetHostedZoneInput{Id: h.Id}, func(options *route53.Options) {
				options.Region = cl.Region
			})
			if err != nil {
				return err
			}

			var delegationSetId *string
			if gotHostedZone.DelegationSet != nil {
				delegationSetId = gotHostedZone.DelegationSet.Id
			}
			res <- &route53HostedZoneWrapper{
				HostedZone:      *gotHostedZone.HostedZone,
				Tags:            client.TagsToMap(getTags(*h.Id, tagsResponse.ResourceTagSets)),
				DelegationSetId: delegationSetId,
				DelegationSet:   gotHostedZone.DelegationSet,
				VPCs:            gotHostedZone.VPCs,
			}
		}
		return nil
	}

	var zones []tableoptions.Route53ListHostedZonesInput
	params := route53.ListHostedZonesInput{}
	for _, filter := range opts.Filters() {
		if filter.Id != nil {
			zones = append(zones, tableoptions.Route53ListHostedZonesInput{
				GetHostedZoneInput: route53.GetHostedZoneInput{Id: filter.Id},
			})
			continue
		}

		paginator := route53.NewListHostedZonesPaginator(svc, &params)
		for paginator.HasMorePages() {
			page, err := paginator.NextPage(ctx, func(options *route53.Options) {
				options.Region = cl.Region
			})
			if err != nil {
				return err
			}

			for _, hz := range page.HostedZones {
				zones = append(zones, tableoptions.Route53ListHostedZonesInput{
					GetHostedZoneInput: route53.GetHostedZoneInput{Id: hz.Id},
				})
			}
		}
	}

	// Process the zones in batches of 10 as that's the maximum number of resource IDs that can be requested in a single call
	// https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListTagsForResources.html#API_ListTagsForResources_RequestBody
	zoneCount := len(zones)
	for i := 0; i < zoneCount; i += maxNumberOfTagResourceIds {
		end := i + maxNumberOfTagResourceIds
		if end > zoneCount {
			end = zoneCount
		}

		if err := processHostedZonesBundle(zones[i:end]); err != nil {
			return err
		}
	}

	return nil
}

func resolveRoute53HostedZoneArn(_ context.Context, meta schema.ClientMeta, resource *schema.Resource, c schema.Column) error {
	cl := meta.(*client.Client)
	hz := resource.Item.(*route53HostedZoneWrapper)
	return resource.Set(c.Name, arn.ARN{
		Partition: cl.Partition,
		Service:   string(client.Route53Service),
		Region:    "",
		AccountID: "",
		Resource:  fmt.Sprintf("hostedzone/%s", aws.ToString(hz.Id)),
	}.String())
}
