// Copyright CloudQuery Authors
// This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0.
// If a copy of the MPL was not distributed with this file, You can obtain one at https://mozilla.org/MPL/2.0/.

package s3

import (
	"context"
	"errors"

	"github.com/apache/arrow-go/v18/arrow"
	"github.com/aws/aws-sdk-go-v2/aws"
	"github.com/aws/aws-sdk-go-v2/service/s3"
	"github.com/cloudquery/cloudquery/plugins/source/aws/client"
	"github.com/cloudquery/cloudquery/plugins/source/aws/resources/services/s3/models"
	"github.com/cloudquery/plugin-sdk/v4/schema"
	"github.com/cloudquery/plugin-sdk/v4/transformers"
)

func Buckets() *schema.Table {
	tableName := "aws_s3_buckets"
	return &schema.Table{
		Name:                tableName,
		PermissionsNeeded:   client.TablePermissions(tableName),
		Resolver:            listS3Buckets,
		PreResourceResolver: resolveS3BucketsAttributes,
		Description:         `https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListBuckets.html`,
		Transform:           transformers.TransformWithStruct(&models.WrappedBucket{}),
		Multiplex:           client.ServiceAccountRegionMultiplexer(tableName, client.AWSServiceS3.String()),
		Columns: []schema.Column{
			client.DefaultAccountIDColumn(false),
			{
				Name:                "arn",
				Type:                arrow.BinaryTypes.String,
				Resolver:            resolveBucketARN(),
				PrimaryKeyComponent: true,
			},
		},

		Relations: []*schema.Table{
			bucketCorsRules(),
			bucketEncryptionRules(),
			bucketGrants(),
			bucketLifecycles(),
			bucketNotificationConfigurations(),
			bucketObjectLockConfigurations(),
			bucketWebsites(),
			bucketLogging(),
			bucketOwnershipControls(),
			bucketReplications(),
			bucketPublicAccessBlock(),
			bucketVersionings(),
			bucketPolicies(),
			objects(),
		},
	}
}

func listS3Buckets(ctx context.Context, meta schema.ClientMeta, _ *schema.Resource, res chan<- any) error {
	cl := meta.(*client.Client)
	svc := cl.Services(client.AWSServiceS3).S3

	paginator := s3.NewListBucketsPaginator(svc, &s3.ListBucketsInput{BucketRegion: aws.String(cl.Region)})

	for paginator.HasMorePages() {
		page, err := paginator.NextPage(ctx, func(o *s3.Options) {
			o.Region = cl.Region
		})
		if err != nil {
			return err
		}
		results := make([]*models.WrappedBucket, len(page.Buckets))
		for i, bucket := range page.Buckets {
			results[i] = &models.WrappedBucket{
				Name:         bucket.Name,
				CreationDate: bucket.CreationDate,
			}
		}
		res <- results
	}

	return nil
}

func resolveS3BucketsAttributes(ctx context.Context, meta schema.ClientMeta, r *schema.Resource) error {
	resource := r.Item.(*models.WrappedBucket)
	cl := meta.(*client.Client)

	resource.Region = cl.Region

	var errAll []error

	resolvers := []func(context.Context, schema.ClientMeta, *models.WrappedBucket) error{
		resolveBucketPolicyStatus,
		resolveBucketTagging,
	}
	for _, resolver := range resolvers {
		if err := resolver(ctx, meta, resource); err != nil {
			// If we received any error other than NoSuchBucketError, we return as this indicates that the bucket has been deleted
			// and therefore no other attributes can be resolved
			if isBucketNotFoundError(cl, err) {
				r.Item = resource
				return errors.Join(errAll...)
			}
			// This enables 403 errors to be recorded, but not block subsequent resolver calls
			errAll = append(errAll, err)
		}
	}
	r.Item = resource
	return errors.Join(errAll...)
}

func resolveBucketPolicyStatus(ctx context.Context, meta schema.ClientMeta, resource *models.WrappedBucket) error {
	cl := meta.(*client.Client)
	svc := cl.Services(client.AWSServiceS3).S3
	policyStatusOutput, err := svc.GetBucketPolicyStatus(ctx, &s3.GetBucketPolicyStatusInput{Bucket: resource.Name}, func(o *s3.Options) {
		o.Region = resource.Region
	})
	// check if we got an error but its access denied we can continue
	if err != nil {
		if client.IsAWSError(err, "NoSuchBucketPolicy") {
			return nil
		}
		if client.IgnoreAccessDeniedServiceDisabled(err) {
			return nil
		}
		return err
	}
	if policyStatusOutput != nil {
		resource.PolicyStatus = policyStatusOutput.PolicyStatus
	}
	return nil
}

func resolveBucketTagging(ctx context.Context, meta schema.ClientMeta, resource *models.WrappedBucket) error {
	cl := meta.(*client.Client)
	svc := cl.Services(client.AWSServiceS3).S3
	taggingOutput, err := svc.GetBucketTagging(ctx, &s3.GetBucketTaggingInput{Bucket: resource.Name}, func(o *s3.Options) {
		o.Region = resource.Region
	})
	if err != nil {
		// If buckets tags are not set it will return an error instead of empty result
		if client.IsAWSError(err, "NoSuchTagSet") {
			return nil
		}
		if client.IgnoreAccessDeniedServiceDisabled(err) {
			return nil
		}
		return err
	}
	if taggingOutput == nil {
		return nil
	}
	tags := make(map[string]*string, len(taggingOutput.TagSet))
	for _, t := range taggingOutput.TagSet {
		tags[*t.Key] = t.Value
	}
	resource.Tags = tags
	return nil
}

func isBucketNotFoundError(cl *client.Client, err error) bool {
	if cl.IsNotFoundError(err) {
		return true
	}
	if err.Error() == "bucket not found" {
		return true
	}
	return false
}

func resolveBucketARN() schema.ColumnResolver {
	return client.ResolveARNGlobal(client.S3Service, func(resource *schema.Resource) ([]string, error) {
		return []string{*resource.Item.(*models.WrappedBucket).Name}, nil
	})
}
