// Copyright CloudQuery Authors
// This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0.
// If a copy of the MPL was not distributed with this file, You can obtain one at https://mozilla.org/MPL/2.0/.

package client

import (
	"context"
	"errors"
	"fmt"
	"slices"
	"strings"

	"github.com/cloudflare/cloudflare-go/v5"
	"github.com/cloudflare/cloudflare-go/v5/accounts"
	"github.com/cloudflare/cloudflare-go/v5/option"
	"github.com/cloudflare/cloudflare-go/v5/zones"

	"github.com/cloudquery/plugin-sdk/v4/schema"
	"github.com/rs/zerolog"
	"github.com/samber/lo"
)

type AccountZones map[string]struct {
	AccountId string
	Zones     []string
}

type Services map[string]CloudflareServices

type Client struct {
	logger        zerolog.Logger
	accountsZones AccountZones
	allServices   Services
	Services      CloudflareServices
	AccountId     string
	ZoneId        string
}

const MaxItemsPerPage = 200

func New(logger zerolog.Logger, allSvcs Services, svcs CloudflareServices, accountsZones AccountZones) Client {
	return Client{
		logger:        logger,
		accountsZones: accountsZones,
		allServices:   allSvcs,
		Services:      svcs,
	}
}

func (c *Client) Logger(ctx context.Context) *zerolog.Logger {
	loggerContext := zerolog.Ctx(ctx).With()
	if c.AccountId != "" {
		loggerContext = loggerContext.Str("account_id", c.AccountId)
	}
	if c.ZoneId != "" {
		loggerContext = loggerContext.Str("zone_id", c.ZoneId)
	}
	return lo.ToPtr(loggerContext.Logger())
}

func (c *Client) ID() string {
	idStrings := []string{
		c.AccountId,
		c.ZoneId,
	}

	return strings.TrimRight(strings.Join(idStrings, ":"), ":")
}

func (c *Client) withAccountID(accountId string) *Client {
	return &Client{
		logger:        c.logger.With().Str("account_id", accountId).Logger(),
		accountsZones: c.accountsZones,
		allServices:   c.allServices,
		Services:      c.allServices[accountId],
		AccountId:     accountId,
	}
}

func (c *Client) withZoneID(accountId, zoneId string) *Client {
	return &Client{
		logger:        c.logger.With().Str("account_id", accountId).Str("zone_id", zoneId).Logger(),
		accountsZones: c.accountsZones,
		allServices:   c.allServices,
		AccountId:     accountId,
		Services:      c.allServices[accountId],
		ZoneId:        zoneId,
	}
}

func Configure(ctx context.Context, logger zerolog.Logger, spec *Spec) (schema.ClientMeta, error) {
	cfClient := getCloudflareClient(spec)
	if cfClient == nil {
		return nil, errors.New("failed to create Cloudflare client v4")
	}
	svcs := NewCloudflareServices(cfClient)

	accountsZones, err := getRelevantAccountsAndZones(ctx, svcs, spec)
	if err != nil {
		return nil, err
	}

	if len(accountsZones) == 0 {
		return nil, errors.New("no accounts found")
	}

	services := make(Services)
	for _, account := range accountsZones {
		c := getCloudflareClient(spec)
		if c == nil {
			return nil, errors.New("failed to create Cloudflare client v4")
		}
		services[account.AccountId] = NewCloudflareServices(c)
	}

	c := New(logger, services, svcs, accountsZones)
	return &c, nil
}

func getCloudflareClient(config *Spec) *cloudflare.Client {
	opts := []option.RequestOption{
		option.WithMaxRetries(10),
	}
	if config.Token != "" {
		opts = append(opts, option.WithAPIToken(config.Token))
	} else {
		opts = append(opts, option.WithAPIEmail(config.ApiEmail), option.WithAPIKey(config.ApiKey))
	}
	return cloudflare.NewClient(opts...)
}

func getRelevantAccountsAndZones(ctx context.Context, svcs CloudflareServices, spec *Spec) (AccountZones, error) {
	var accountIDs []string

	if len(spec.Accounts) == 0 {
		page, err := svcs.AccountsAccountService.List(ctx, accounts.AccountListParams{PerPage: cloudflare.Float(50)})
		if err != nil {
			return nil, fmt.Errorf("failed to list accounts: %w", err)
		}

		var accs []accounts.Account
		for page != nil {
			accs = append(accs, page.Result...)
			page, err = page.GetNextPage()
			if err != nil {
				return nil, fmt.Errorf("failed to get next page for accounts: %w", err)
			}
		}
		relevantAccounts := lo.Filter(accs, func(a accounts.Account, _ int) bool {
			return len(spec.Accounts) == 0 || slices.Contains(spec.Accounts, a.ID)
		})
		accountIDs = lo.Map(relevantAccounts, func(a accounts.Account, _ int) string { return a.ID })
	} else {
		accountIDs = spec.Accounts
	}

	accountZones := make(AccountZones, len(accountIDs))
	for _, accountID := range accountIDs {
		page, err := svcs.ZonesZoneService.List(ctx, zones.ZoneListParams{
			Account: cloudflare.F(zones.ZoneListParamsAccount{
				ID: cloudflare.F(accountID),
			}),
			PerPage: cloudflare.Float(50),
		})
		if err != nil {
			return nil, fmt.Errorf("failed to list zones: %w", err)
		}
		var collectedZones []zones.Zone
		for page != nil {
			collectedZones = append(collectedZones, page.Result...)
			page, err = page.GetNextPage()
			if err != nil {
				return nil, fmt.Errorf("failed to get next page for zones: %w", err)
			}
		}

		accountZones[accountID] = struct {
			AccountId string
			Zones     []string
		}{
			AccountId: accountID,
			Zones:     lo.Map(collectedZones, func(z zones.Zone, _ int) string { return z.ID }),
		}
	}
	return accountZones, nil
}
