// Copyright CloudQuery Authors
// This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0.
// If a copy of the MPL was not distributed with this file, You can obtain one at https://mozilla.org/MPL/2.0/.

package access_applications

import (
	"context"
	"fmt"

	"github.com/apache/arrow-go/v18/arrow"
	cf "github.com/cloudflare/cloudflare-go/v5"
	"github.com/cloudflare/cloudflare-go/v5/zero_trust"
	"github.com/cloudquery/cloudquery/plugins/source/cloudflare/client"
	"github.com/cloudquery/plugin-sdk/v4/schema"
)

func AccessApplications() *schema.Table {
	return &schema.Table{
		Name:              "cloudflare_access_applications",
		Resolver:          fetchAccessApplications,
		PermissionsNeeded: []string{"Access: Apps and Policies Read"},
		Multiplex:         client.ZoneMultiplex,
		Transform:         client.TransformWithStruct(&zero_trust.AccessApplicationListResponse{}),
		Columns: []schema.Column{
			{
				Name:        "account_id",
				Type:        arrow.BinaryTypes.String,
				Resolver:    client.ResolveAccountID,
				Description: `The Account ID of the resource.`,
			},
			{
				Name:        "zone_id",
				Type:        arrow.BinaryTypes.String,
				Resolver:    client.ResolveZoneID,
				Description: `Zone identifier tag.`,
			},
			{
				Name:       "id",
				Type:       arrow.BinaryTypes.String,
				Resolver:   schema.PathResolver("ID"),
				PrimaryKey: true,
			},
		},
		Relations: []*schema.Table{
			selfHostedDomainsTable(),
		},
	}
}
func fetchAccessApplications(ctx context.Context, meta schema.ClientMeta, parent *schema.Resource, res chan<- any) error {
	c := meta.(*client.Client)

	params := zero_trust.AccessApplicationListParams{
		ZoneID: cf.F(c.ZoneId),
	}
	page, err := c.Services.ZeroTrustAccessApplicationService.List(ctx, params)
	if err != nil {
		return fmt.Errorf("failed to list access applications: %w", err)
	}
	for page != nil {
		res <- page.Result
		page, err = page.GetNextPage()
		if err != nil {
			return fmt.Errorf("failed to get next page for access applications: %w", err)
		}
	}
	return nil
}
