// Copyright CloudQuery Authors
// This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0.
// If a copy of the MPL was not distributed with this file, You can obtain one at https://mozilla.org/MPL/2.0/.

package access_applications

import (
	"context"

	"github.com/cloudflare/cloudflare-go/v5/zero_trust"
	"github.com/cloudquery/plugin-sdk/v4/schema"
	"github.com/cloudquery/plugin-sdk/v4/transformers"
)

type AccessApplicationsSelfHostedDomains struct {
	ApplicationID string `json:"application_id"`
	Domain        string `json:"domain"`
}

func selfHostedDomainsTable() *schema.Table {
	return &schema.Table{
		Name:              "cloudflare_access_applications_self_hosted_domains",
		PermissionsNeeded: []string{"Access: Apps and Policies Read"},
		Transform:         transformers.TransformWithStruct(&AccessApplicationsSelfHostedDomains{}, transformers.WithPrimaryKeys("ApplicationID", "Domain")),
		Resolver:          fetchSelfHostedDomains,
	}
}

func fetchSelfHostedDomains(ctx context.Context, meta schema.ClientMeta, parent *schema.Resource, res chan<- any) error {
	application := parent.Item.(zero_trust.AccessApplicationListResponse)

	domains, ok := application.SelfHostedDomains.([]zero_trust.SelfHostedDomains)
	if !ok {
		return nil
	}
	for _, domain := range domains {
		row := AccessApplicationsSelfHostedDomains{
			ApplicationID: application.ID,
			Domain:        domain,
		}
		res <- row
	}

	return nil
}
