// Copyright CloudQuery Authors
// This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0.
// If a copy of the MPL was not distributed with this file, You can obtain one at https://mozilla.org/MPL/2.0/.

package client

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"fmt"

	"cloud.google.com/go/firestore"
	"github.com/cloudquery/plugin-sdk/v4/plugin"
	"github.com/cloudquery/plugin-sdk/v4/premium"
	"github.com/cloudquery/plugin-sdk/v4/schema"
	"github.com/rs/zerolog"
	"google.golang.org/api/option"
)

type Client struct {
	plugin.UnimplementedDestination
	logger      zerolog.Logger
	tables      schema.Tables
	options     plugin.NewClientOptions
	client      *firestore.Client
	usage       premium.UsageClient
	spec        *Spec
	isSpecValid bool
}

var _ schema.ClientMeta = (*Client)(nil)

func (*Client) ID() string {
	return "source-firestore"
}

func Configure(ctx context.Context, logger zerolog.Logger, spec []byte, opts plugin.NewClientOptions) (plugin.Client, error) {
	if opts.NoConnection && (spec == nil || bytes.Equal(spec, []byte("null"))) {
		return &Client{
			logger:  logger,
			options: opts,
			tables:  schema.Tables{},
		}, nil
	}
	var firestoreSpec Spec
	err := json.Unmarshal(spec, &firestoreSpec)
	if err != nil {
		return nil, fmt.Errorf("failed to unmarshal spec: %w", err)
	}

	firestoreSpec.SetDefaults()
	if err := firestoreSpec.Validate(); err != nil {
		return nil, fmt.Errorf("failed to validate spec: %w", err)
	}

	var client *firestore.Client
	if firestoreSpec.ServiceAccountJSON == "" {
		// Use Application Default Credentials
		client, err = firestore.NewClient(ctx, firestoreSpec.ProjectID)
		if err != nil {
			return nil, fmt.Errorf("failed to create firestore client: %w", err)
		}
	} else {
		creds := option.WithCredentialsJSON([]byte(firestoreSpec.ServiceAccountJSON))
		client, err = firestore.NewClient(ctx, firestoreSpec.ProjectID, creds)
		if err != nil {
			return nil, fmt.Errorf("failed to create firestore client: %w", err)
		}
	}

	c := &Client{
		logger:      logger.With().Str("module", "firestore-source").Logger(),
		client:      client,
		spec:        &firestoreSpec,
		options:     opts,
		isSpecValid: true,
	}
	c.usage, err = premium.NewUsageClient(opts.PluginMeta, premium.WithLogger(logger))
	if err != nil {
		return nil, fmt.Errorf("failed to initialize usage client: %w", err)
	}

	c.tables, err = c.listTables(ctx, client)
	if err != nil {
		return nil, fmt.Errorf("failed to list tables: %w", err)
	}
	if len(c.tables) == 0 {
		return nil, errors.New("no tables found")
	}

	return c, nil
}

func (c Client) Tables(ctx context.Context, opts plugin.TableOptions) (schema.Tables, error) {
	if !c.isSpecValid {
		return schema.Tables{}, nil
	}
	return c.tables.FilterDfs(opts.Tables, opts.SkipTables, opts.SkipDependentTables)
}

func (c Client) Close(_ context.Context) error {
	return c.client.Close()
}
