// Copyright CloudQuery Authors
// This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0.
// If a copy of the MPL was not distributed with this file, You can obtain one at https://mozilla.org/MPL/2.0/.

package client

import (
	_ "embed"
	"encoding/base64"
	"fmt"
	"strings"

	"github.com/invopop/jsonschema"

	"cloud.google.com/go/firestore"
)

// Spec is the (nested) spec used by Firestore Source Plugin
type Spec struct {
	// The ID of the project to use for this client. If not specified, the project id will be auto-detected from the credentials.
	ProjectID string `json:"project_id"`
	// If `true` the `service_account_json` content will be treated as base64-encoded.
	UseBase64 bool `json:"use_base64" jsonschema:"default=false"`
	//  Service account JSON content.
	ServiceAccountJSON string `json:"service_account_json"`

	// Amount of rows to be packed into a single Apache Arrow record to be sent over the wire during sync.
	RowsPerRecord int `json:"rows_per_record,omitempty" jsonschema:"minimum=1,default=500"`

	// Maximum batch size for each request when reading Firestore data.
	MaxBatchSize int `json:"max_batch_size" jsonschema:"minimum=1,default=50000"`
	// List of fields to order the results by.
	OrderBy string `json:"order_by"`
	// The order direction used when `order_by` is `true`.
	OrderDirection string `json:"order_direction" jsonschema:"enum=asc,enum=desc,default=asc"`
}

func (s *Spec) Validate() error {
	// decode base64 if needed - note if the Validate function is removed from the spec, this will need to be done
	// elsewhere in the application
	if s.UseBase64 {
		data, err := base64.StdEncoding.DecodeString(s.ServiceAccountJSON)
		if err != nil {
			return fmt.Errorf("failed to decode service_account_json: %w", err)
		}
		s.ServiceAccountJSON = string(data)
	}

	s.OrderDirection = strings.ToLower(s.OrderDirection)
	if s.OrderDirection != "" && s.OrderDirection != "asc" && s.OrderDirection != "desc" {
		return fmt.Errorf("invalid order_by_direction %s", s.OrderDirection)
	}

	return nil
}

func (Spec) JSONSchemaExtend(sc *jsonschema.Schema) {
	// Configure defaults
	sc.Properties.Value("project_id").Default = firestore.DetectProjectID
}

func (s *Spec) SetDefaults() {
	if s.RowsPerRecord < 1 {
		s.MaxBatchSize = 500
	}
	if s.MaxBatchSize < 1 {
		s.MaxBatchSize = 50000
	}
	if s.ProjectID == "" {
		s.ProjectID = firestore.DetectProjectID
	}
}

//go:embed schema.json
var JSONSchema string
