// Copyright CloudQuery Authors
// This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0.
// If a copy of the MPL was not distributed with this file, You can obtain one at https://mozilla.org/MPL/2.0/.

package client

import (
	"reflect"
	"slices"
	"strings"

	"google.golang.org/api/googleapi"
	"google.golang.org/protobuf/reflect/protoreflect"
	"google.golang.org/protobuf/types/known/durationpb"
	"google.golang.org/protobuf/types/known/timestamppb"

	"github.com/apache/arrow-go/v18/arrow"
	"github.com/cloudquery/plugin-sdk/v4/schema"
	"github.com/cloudquery/plugin-sdk/v4/transformers"
	cqtypes "github.com/cloudquery/plugin-sdk/v4/types"

	"cloud.google.com/go/aiplatform/apiv1/aiplatformpb"
	"cloud.google.com/go/container/apiv1/containerpb"
	"google.golang.org/protobuf/types/known/structpb"
)

var toReplace = map[string]string{
	"c_d_n":   "cdn",
	"i_p_":    "ip_",
	"ipv_6":   "ipv6",
	"i_pv_4":  "ipv4",
	"i_pv4":   "ipv4",
	"oauth_2": "oauth2",
	"c_o_r_s": "cors",
	"r_p_o":   "rpo",
}

func replaceTransformer(field reflect.StructField) (string, error) {
	name, err := transformers.DefaultNameTransformer(field)
	if err != nil {
		return "", err
	}
	for k, v := range toReplace {
		name = strings.ReplaceAll(name, k, v)
	}
	return name, nil
}

func DefaultTypeTransformer(field reflect.StructField) (arrow.DataType, error) {
	switch reflect.New(field.Type).Elem().Interface().(type) {
	case *timestamppb.Timestamp,
		timestamppb.Timestamp:
		return arrow.FixedWidthTypes.Timestamp_us, nil
	case *durationpb.Duration,
		durationpb.Duration:
		return arrow.PrimitiveTypes.Int64, nil
	case googleapi.RawMessage:
		return cqtypes.NewJSONType(), nil
	case protoreflect.Enum:
		return arrow.BinaryTypes.String, nil
	case nil:
		return cqtypes.NewJSONType(), nil
	default:
		return nil, nil
	}
}

func resolverTransformer(field reflect.StructField, path string) schema.ColumnResolver {
	switch reflect.New(field.Type).Elem().Interface().(type) {
	case *timestamppb.Timestamp:
		return ResolveProtoTimestamp(path)
	case *durationpb.Duration:
		return ResolveProtoDuration(path)
	case protoreflect.Enum:
		return ResolveProtoEnum(path)
	case googleapi.RawMessage:
		return ResolveGoogleAPIRawMessage(path)
	default:
		return nil
	}
}

func ignoreInTestsTransformer(field reflect.StructField) bool {
	switch field.Type {
	case reflect.TypeOf(structpb.Value{}),
		reflect.TypeOf(&structpb.Value{}),
		reflect.TypeOf(structpb.Struct{}),
		reflect.TypeOf(&structpb.Struct{}),
		reflect.TypeOf(aiplatformpb.Model{}),
		reflect.TypeOf(&aiplatformpb.Model{}),
		reflect.TypeOf(aiplatformpb.PipelineJob_RuntimeConfig{}),
		reflect.TypeOf(&aiplatformpb.PipelineJob_RuntimeConfig{}),
		reflect.TypeOf(&containerpb.AnonymousAuthenticationConfig{}):
		return true
	default:
		return false
	}
}

var options = []transformers.StructTransformerOption{
	transformers.WithNameTransformer(replaceTransformer),
	transformers.WithTypeTransformer(DefaultTypeTransformer),
	transformers.WithResolverTransformer(resolverTransformer),
	transformers.WithIgnoreInTestsTransformer(ignoreInTestsTransformer),
}

func TransformWithStruct(t any, opts ...transformers.StructTransformerOption) schema.Transform {
	return transformers.TransformWithStruct(t, append(options, opts...)...)
}

func FieldsToJSONTransformer(fields ...string) func(sf reflect.StructField) (arrow.DataType, error) {
	return func(sf reflect.StructField) (arrow.DataType, error) {
		if slices.Contains(fields, sf.Name) {
			return cqtypes.ExtensionTypes.JSON, nil
		}
		return DefaultTypeTransformer(sf)
	}
}

// SliceToMapResolverTransformer adds possibility to override the column resolver for provided fields
func SliceToMapResolverTransformer(fields ...string) func(field reflect.StructField, path string) schema.ColumnResolver {
	return func(field reflect.StructField, path string) schema.ColumnResolver {
		if slices.Contains(fields, path) {
			return ResolveSliceToMap(path)
		}
		return resolverTransformer(field, path)
	}
}
