// Copyright CloudQuery Authors
// This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0.
// If a copy of the MPL was not distributed with this file, You can obtain one at https://mozilla.org/MPL/2.0/.

package plugin

import (
	"encoding/json"
	"reflect"
	"regexp"
	"strings"

	"github.com/cloudquery/cloudquery/plugins/source/gcp/client"
	"github.com/cloudquery/cloudquery/plugins/source/gcp/client/spec"
	"github.com/cloudquery/cloudquery/plugins/source/gcp/client/spec/tableoptions"
	schemaDocs "github.com/cloudquery/codegen/jsonschema/docs"
	"github.com/cloudquery/plugin-sdk/v4/caser"
	"github.com/cloudquery/plugin-sdk/v4/docs"
	"github.com/cloudquery/plugin-sdk/v4/plugin"
	"github.com/cloudquery/plugin-sdk/v4/schema"
	invoschema "github.com/invopop/jsonschema"
	"golang.org/x/exp/maps"
)

var (
	Name    = "gcp"
	Kind    = "source"
	Team    = "cloudquery"
	Version = "development"
)

var gcpExceptions = map[string]string{
	"accessapproval":       "Access Approval",
	"aiplatform":           "AI Platform",
	"apigateway":           "API Gateway",
	"apikeys":              "API Keys",
	"appengine":            "App Engine",
	"artifactregistry":     "Artifact Registry",
	"baremetalsolution":    "Bare Metal Solution",
	"bigquery":             "BigQuery",
	"billingbudgets":       "Billing Budgets",
	"binaryauthorization":  "Binary Authorization",
	"cloudasset":           "Cloud Asset",
	"cloudbuild":           "Cloud Build",
	"cloudchannel":         "Cloud Channel",
	"clouddms":             "Cloud DMS",
	"cloudfunctions":       "Cloud Functions",
	"cloudidentity":        "Cloud Identity",
	"cloudkms":             "Cloud KMS",
	"cloudresourcemanager": "Cloud Resource Manager",
	"cloudscheduler":       "Cloud Scheduler",
	"cloudtasks":           "Cloud Tasks",
	"cloudtrace":           "Cloud Trace",
	"composer":             "Composer",
	"compute":              "Compute",
	"container":            "Container",
	"containeranalysis":    "Container Analysis",
	"dataproc":             "Dataproc",
	"dataprocmetastore":    "Dataproc Metastore",
	"datashare":            "Datashare",
	"deploymentmanager":    "Deployment Manager",
	"dialogflow":           "Dialogflow",
	"dlp":                  "DLP",
	"domains":              "Domains",
	"ekm":                  "Cloud External Key Manager (EKM)",
	"featurestore":         "Feature Store",
	"featurestores":        "featurestores",
	"gameservices":         "Game Services",
	"grpc":                 "gRPC",
	"http":                 "HTTP",
	"https":                "HTTPs",
	"iamcredentials":       "IAM Credentials",
	"iap":                  "IAP",
	"identityplatform":     "Identity Platform",
	"indexendpoint":        "Index Endpoint",
	"iot":                  "IoT",
	"kms":                  "Cloud Key Management Service (KMS)",
	"memcache":             "Memcache",
	"ml":                   "ML",
	"networkmanagement":    "Network Management",
	"networkconnectivity":  "Network Connectivity",
	"networkservices":      "Network Services",
	"nfs":                  "NFS",
	"osconfig":             "OS Config",
	"oslogin":              "OS Login",
	"policyanalyzer":       "Policy Analyzer",
	"privateca":            "Private CA",
	"pubsub":               "PubSub",
	"secretmanager":        "Secret Manager",
	"specialistpool":       "Specialist Pool",
	"sql":                  "SQL",
	"ssl":                  "SSL",
	"tensorboard":          "TensorBoard",
	"videotranscoder":      "Video Transcoder",
	"vm":                   "VM",
	"vmmigration":          "VM Migration",
	"vms":                  "Virtual Machines (VMs)",
	"vpcaccess":            "VPC Access",
	"websecurityscanner":   "Web Security Scanner",
}

func titleTransformer(table *schema.Table) error {
	if table.Title != "" {
		return nil
	}
	exceptions := maps.Clone(docs.DefaultTitleExceptions)
	for k, v := range gcpExceptions {
		exceptions[k] = v
	}
	csr := caser.New(caser.WithCustomExceptions(exceptions))
	t := csr.ToTitle(table.Name)
	table.Title = strings.Trim(strings.ReplaceAll(t, "  ", " "), " ")
	return nil
}

func descriptionTransformer() func(*schema.Table) error {
	var sc invoschema.Schema
	if err := json.Unmarshal([]byte(spec.JSONSchema), &sc); err != nil {
		return nil
	}

	tableNamesToOptionsDocs := make(map[string]string)
	tableOptionsType := reflect.ValueOf(&tableoptions.TableOptions{}).Elem().Type()
	for i := range tableOptionsType.NumField() {
		field := tableOptionsType.Field(i)
		fieldType := strings.Split(field.Type.String(), ".")[1]
		defValue, ok := sc.Definitions[fieldType]
		if !ok {
			panic("definition not found for " + field.Name)
		}
		tableName := strings.Split(field.Tag.Get("json"), ",")[0]
		if tableName == "" {
			panic("table name not found for " + field.Name)
		}
		newRoot := sc
		newRoot.ID = "Table Options"
		newRoot.Ref = "#/$defs/" + "Table Options"
		newRoot.Definitions["Table Options"] = defValue
		sch, _ := json.Marshal(newRoot)
		doc, _ := schemaDocs.Generate(sch, 1)
		tocRegex := regexp.MustCompile(`# Table of contents[\s\S]+?##`)
		tableNamesToOptionsDocs[tableName] = tocRegex.ReplaceAllString(doc, "##")
	}

	return func(table *schema.Table) error {
		if tableNamesToOptionsDocs[table.Name] != "" {
			table.Description = table.Description + "\n\n" + tableNamesToOptionsDocs[table.Name]
		}
		return nil
	}
}

func Plugin() *plugin.Plugin {
	return plugin.NewPlugin(
		Name,
		Version,
		NewClient,
		plugin.WithBuildTargets(buildTargets()),
		plugin.WithJSONSchema(spec.JSONSchema),
		plugin.WithKind(Kind),
		plugin.WithTeam(Team),
		plugin.WithConnectionTester(client.NewConnectionTester(client.New)),
	)
}
