// Copyright CloudQuery Authors
// This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0.
// If a copy of the MPL was not distributed with this file, You can obtain one at https://mozilla.org/MPL/2.0/.

package compute

import (
	"context"
	"strings"

	compute "cloud.google.com/go/compute/apiv1"
	pb "cloud.google.com/go/compute/apiv1/computepb"
	"github.com/cloudquery/cloudquery/plugins/source/gcp/client"
	"github.com/cloudquery/cloudquery/plugins/source/gcp/client/permissions"
	"github.com/cloudquery/plugin-sdk/v4/schema"
	"github.com/cloudquery/plugin-sdk/v4/transformers"
)

func interconnectAttachments() *schema.Table {
	return &schema.Table{
		Name:              "gcp_compute_interconnect_attachments",
		Description:       "https://cloud.google.com/compute/docs/reference/rest/v1/interconnectAttachments#InterconnectAttachment",
		Resolver:          fetchInterconnectAttachments,
		PermissionsNeeded: permissions.NewList(permissions.ComputeInterconnectAttachmentsList),
		Transform:         client.TransformWithStruct(&pb.InterconnectAttachment{}, transformers.WithPrimaryKeys("SelfLink")),
		Columns: []schema.Column{
			client.ProjectIDColumn(false),
		},
	}
}

func fetchInterconnectAttachments(ctx context.Context, meta schema.ClientMeta, parent *schema.Resource, res chan<- any) error {
	c := meta.(*client.Client)
	location := parent.Item.(*pb.InterconnectLocation)

	gcpClient, err := compute.NewInterconnectAttachmentsRESTClient(ctx, c.ClientOptions...)
	if err != nil {
		return err
	}
	for _, regionInfo := range location.GetRegionInfos() {
		regionParts := strings.SplitN(regionInfo.GetRegion(), "/regions/", 2)
		if len(regionParts) != 2 {
			c.Logger(ctx).Error().Any("region_info", regionInfo.GetRegion()).Msg(`region value doesn't contain "/regions/ substring"`)
			continue
		}

		req := &pb.ListInterconnectAttachmentsRequest{
			Project: c.ProjectId,
			Region:  regionParts[1],
		}
		it := gcpClient.List(ctx, req, c.CallOptions...)
		if err := client.Iterate(ctx, it, c, res); err != nil {
			return err
		}
	}
	return nil
}
