// Copyright CloudQuery Authors
// This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0.
// If a copy of the MPL was not distributed with this file, You can obtain one at https://mozilla.org/MPL/2.0/.

package compute

import (
	"context"

	compute "cloud.google.com/go/compute/apiv1"
	"cloud.google.com/go/compute/apiv1/computepb"

	"github.com/cloudquery/cloudquery/plugins/source/gcp/client"
	"github.com/cloudquery/cloudquery/plugins/source/gcp/client/permissions"
	"github.com/cloudquery/plugin-sdk/v4/schema"
	"github.com/cloudquery/plugin-sdk/v4/transformers"
)

func Routes() *schema.Table {
	return &schema.Table{
		Name:              "gcp_compute_routes",
		Description:       `https://cloud.google.com/compute/docs/reference/rest/v1/routes/list#response-body`,
		Resolver:          fetchRoutes,
		PermissionsNeeded: permissions.NewList(permissions.ComputeRoutesList),
		Multiplex:         client.ProjectMultiplexEnabledServices("compute.googleapis.com"),
		Transform: client.TransformWithStruct(
			&computepb.Route{},
			transformers.WithPrimaryKeys("SelfLink"),
			transformers.WithTypeTransformer(client.FieldsToJSONTransformer("Tags")),
			transformers.WithResolverTransformer(client.SliceToMapResolverTransformer("Tags")),
		),
		Columns: []schema.Column{
			client.ProjectIDColumn(false),
		},
	}
}

func fetchRoutes(ctx context.Context, meta schema.ClientMeta, parent *schema.Resource, res chan<- any) error {
	c := meta.(*client.Client)
	req := &computepb.ListRoutesRequest{
		Project: c.ProjectId,
	}
	gcpClient, err := compute.NewRoutesRESTClient(ctx, c.ClientOptions...)
	if err != nil {
		return err
	}

	it := gcpClient.List(ctx, req, c.CallOptions...)
	return client.Iterate(ctx, it, c, res)
}
