// Copyright CloudQuery Authors
// This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0.
// If a copy of the MPL was not distributed with this file, You can obtain one at https://mozilla.org/MPL/2.0/.

package compute

import (
	"context"

	compute "cloud.google.com/go/compute/apiv1"
	pb "cloud.google.com/go/compute/apiv1/computepb"
	"google.golang.org/api/iterator"

	"github.com/cloudquery/cloudquery/plugins/source/gcp/client"
	"github.com/cloudquery/cloudquery/plugins/source/gcp/client/permissions"
	"github.com/cloudquery/plugin-sdk/v4/schema"
	"github.com/cloudquery/plugin-sdk/v4/transformers"
)

func VpnTunnels() *schema.Table {
	return &schema.Table{
		Name:              "gcp_compute_vpn_tunnels",
		Description:       "https://cloud.google.com/compute/docs/reference/rest/v1/vpnTunnels#VpnTunnel",
		Resolver:          fetchVpnTunnels,
		PermissionsNeeded: permissions.NewList(permissions.ComputeVpnTunnelsList),
		Multiplex:         client.ProjectMultiplexEnabledServices("compute.googleapis.com"),
		Transform:         client.TransformWithStruct(&pb.VpnTunnel{}, transformers.WithPrimaryKeys("SelfLink")),
		Columns: []schema.Column{
			client.ProjectIDColumn(false),
		},
	}
}

func fetchVpnTunnels(ctx context.Context, meta schema.ClientMeta, parent *schema.Resource, res chan<- any) error {
	c := meta.(*client.Client)
	req := &pb.AggregatedListVpnTunnelsRequest{
		Project: c.ProjectId,
	}
	gcpClient, err := compute.NewVpnTunnelsRESTClient(ctx, c.ClientOptions...)
	if err != nil {
		return err
	}
	it := gcpClient.AggregatedList(ctx, req, c.CallOptions...)
	for {
		resp, err := it.Next()
		if err == iterator.Done {
			break
		}
		if err != nil {
			return err
		}

		res <- resp.Value.GetVpnTunnels()
	}
	return nil
}
