// Copyright CloudQuery Authors
// This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0.
// If a copy of the MPL was not distributed with this file, You can obtain one at https://mozilla.org/MPL/2.0/.

package iam

import (
	"context"

	"google.golang.org/api/option"

	pb "cloud.google.com/go/iam/admin/apiv1/adminpb"

	"github.com/cloudquery/cloudquery/plugins/source/gcp/client"
	"github.com/cloudquery/cloudquery/plugins/source/gcp/client/permissions"
	"github.com/cloudquery/plugin-sdk/v4/schema"
	"github.com/cloudquery/plugin-sdk/v4/transformers"

	admin "cloud.google.com/go/iam/admin/apiv1"
)

func ServiceAccounts() *schema.Table {
	return &schema.Table{
		Name:              "gcp_iam_service_accounts",
		Description:       `https://cloud.google.com/iam/docs/reference/rest/v1/projects.serviceAccounts#ServiceAccount`,
		Resolver:          fetchServiceAccounts,
		PermissionsNeeded: permissions.NewList(permissions.IamServiceAccountsList),
		Multiplex:         client.ProjectMultiplexEnabledServices("iam.googleapis.com"),
		Transform:         client.TransformWithStruct(&pb.ServiceAccount{}, transformers.WithPrimaryKeys("Name", "UniqueId")),
		Columns:           schema.ColumnList{client.ProjectIDColumn(false)},
		Relations: []*schema.Table{
			serviceAccountKeys(),
			serviceAccountPolicies(),
		},
	}
}

func fetchServiceAccounts(ctx context.Context, meta schema.ClientMeta, parent *schema.Resource, res chan<- any) error {
	c := meta.(*client.Client)
	req := &pb.ListServiceAccountsRequest{
		Name: "projects/" + c.ProjectId,
	}

	options := make([]option.ClientOption, len(c.ClientOptions))
	copy(options, c.ClientOptions)
	if c.TestingGRPCEndpoint != nil {
		options = append(options, option.WithEndpoint(*c.TestingGRPCEndpoint))
	}

	gcpClient, err := admin.NewIamClient(ctx, options...)
	if err != nil {
		return err
	}
	it := gcpClient.ListServiceAccounts(ctx, req, c.CallOptions...)
	return client.Iterate(ctx, it, c, res)
}
