// Copyright CloudQuery Authors
// This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0.
// If a copy of the MPL was not distributed with this file, You can obtain one at https://mozilla.org/MPL/2.0/.

package iam

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"testing"

	"github.com/cloudquery/cloudquery/plugins/source/gcp/client"
	"github.com/cloudquery/plugin-sdk/v4/faker"
	"github.com/julienschmidt/httprouter"
	resourcepb "google.golang.org/api/cloudresourcemanager/v3"
	"google.golang.org/grpc"

	pb "cloud.google.com/go/iam/admin/apiv1/adminpb"
)

func createServiceAccounts(gsrv *grpc.Server) error {
	fakeServer := &fakeServiceAccountsServer{}
	pb.RegisterIAMServer(gsrv, fakeServer)
	return nil
}

type fakeServiceAccountsServer struct {
	pb.UnimplementedIAMServer
}

func (*fakeServiceAccountsServer) ListServiceAccounts(context.Context, *pb.ListServiceAccountsRequest) (*pb.ListServiceAccountsResponse, error) {
	resp := pb.ListServiceAccountsResponse{}
	if err := faker.FakeObject(&resp); err != nil {
		return nil, fmt.Errorf("failed to fake data: %w", err)
	}
	resp.NextPageToken = ""
	return &resp, nil
}

func (*fakeServiceAccountsServer) ListServiceAccountKeys(context.Context, *pb.ListServiceAccountKeysRequest) (*pb.ListServiceAccountKeysResponse, error) {
	resp := pb.ListServiceAccountKeysResponse{}
	if err := faker.FakeObject(&resp); err != nil {
		return nil, fmt.Errorf("failed to fake data: %w", err)
	}
	return &resp, nil
}

func createProjectPolicies(mux *httprouter.Router) error {
	var item resourcepb.Policy
	if err := faker.FakeObject(&item); err != nil {
		return err
	}

	mux.POST("/v3/projects/testProject/serviceAccounts/test string:getIamPolicy", func(w http.ResponseWriter, r *http.Request, _ httprouter.Params) {
		b, err := json.Marshal(&item)
		if err != nil {
			http.Error(w, "unable to marshal request: "+err.Error(), http.StatusBadRequest)
			return
		}
		if _, err := w.Write(b); err != nil {
			http.Error(w, "failed to write", http.StatusBadRequest)
			return
		}
	})
	return nil
}

func TestServiceAccounts(t *testing.T) {
	client.MockTestHelper(t, ServiceAccounts(), client.WithCreateGrpcService(createServiceAccounts), client.WithCreateHTTPServer(createProjectPolicies))
}
