// Copyright CloudQuery Authors
// This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0.
// If a copy of the MPL was not distributed with this file, You can obtain one at https://mozilla.org/MPL/2.0/.

package client

import (
	_ "embed"
	"errors"
	"fmt"
	"os"
	"time"

	"github.com/cloudquery/plugin-sdk/v4/configtype"
	"github.com/cloudquery/plugin-sdk/v4/scheduler"
	"github.com/invopop/jsonschema"
	"github.com/tj/go-naturaldate"
)

// Spec is the (nested) spec used by GitHub Source Plugin
type Spec struct {
	// Personal Access Token, required if not using App Authentication.
	AccessToken string `json:"access_token" jsonschema:"minLength=1"`
	// List of organizations to sync from. You must specify either orgs or repos in the configuration.
	Orgs []string `json:"orgs" jsonschema:"minItems=1"`
	// List of repositories to sync from. The format is owner/repo (e.g. cloudquery/cloudquery).
	// You must specify either orgs or repos in the configuration.
	Repos              []string            `json:"repos" jsonschema:"minItems=1,minLength=1,pattern=^[a-zA-Z0-9-_]+/[a-zA-Z0-9-_]+$"`
	AppAuth            []AppAuthSpec       `json:"app_auth" jsonschema:"minItems=1"`
	EnterpriseSettings *EnterpriseSettings `json:"enterprise"`

	// A best effort maximum number of Go routines to use. Lower this number to reduce memory usage or to avoid hitting GitHub API rate limits.
	Concurrency int `json:"concurrency,omitempty" jsonschema:"default=1500"`

	// The scheduler to use when determining the priority of resources to sync. By default, it is set to `dfs`.
	//
	// For more information about this, see [performance tuning](/docs/advanced-topics/performance-tuning).
	Scheduler *scheduler.Strategy `json:"scheduler,omitempty" jsonschema:"default=dfs"`

	// Controls the number of parallel requests to GitHub when discovering repositories, a negative value means unlimited.
	DiscoveryConcurrency int `json:"discovery_concurrency,omitempty" jsonschema:"default=1"`
	// Include archived repositories when discovering repositories.
	IncludeArchivedRepos bool `json:"include_archived_repos,omitempty"`
	// Path to a local directory that will hold the cache. If set, the plugin will cache the GitHub API responses in this directory. Defaults to an empty string (no cache)
	LocalCachePath string `json:"local_cache_path,omitempty"`

	// Table options to set for specific tables.
	TableOptions TableOptions `json:"table_options,omitempty"`

	// If true, the plugin will add _cq_client_id column to all tables.
	AddCQClientID bool `json:"add_cq_client_id,omitempty"`
}

type TableOptions struct {
	// Table options for the github_workflow_runs table.
	WorkflowRuns WorkflowRunsOptions `json:"github_workflow_runs,omitempty"`

	// Table options for the github_issues table.
	Issues IssuesOptions `json:"github_issues,omitempty"`

	// Table options for the github_repositories table.
	Repositories RepositoriesOptions `json:"github_repositories,omitempty"`

	// Table options for the repository_commits table.
	RepositoryCommits RepositoryCommitsOptions `json:"github_repository_commits,omitempty"`

	// Table options for the github_copilot_org_metrics table.
	CopilotOrgMetrics CopilotOrgMetrics `json:"github_copilot_org_metrics,omitempty"`

	// Table options for the github_copilot_team_metrics table.
	CopilotTeamMetrics CopilotTeamMetrics `json:"github_copilot_team_metrics,omitempty"`

	// Table options for the github_organization_dependabot_alerts table.
	OrgDependabotAlerts OrgDependabotAlerts `json:"github_organization_dependabot_alerts,omitempty"`

	// Table options for the github_repository_dependabot_alerts table.
	RepositoryDependabotAlerts RepositoryDependabotAlerts `json:"github_repository_dependabot_alerts,omitempty"`

	// Table options for the github_code_scanning_alerts table.
	CodeScanningAlerts CodeScanningAlertsOptions `json:"github_code_scanning_alerts,omitempty"`

	// Table options for the github_pull_requests table.
	PullRequests PullRequestsOptions `json:"github_pull_requests,omitempty"`

	// Table options for the github_pull_request_closing_issues_references table.
	PullRequestClosingIssuesReferences PullRequestClosingIssuesReferencesOptions `json:"github_pull_request_closing_issues_references,omitempty"`

	// Table options for the github_secret_scanning_alerts table.
	SecretScanningAlerts SecretScanningAlertsOptions `json:"github_secret_scanning_alerts,omitempty"`
}

type PullRequestClosingIssuesReferencesOptions struct {
	UpdatedAfter configtype.Time `json:"updated_after,omitempty"`
}

type PullRequestsOptions struct {
	UpdatedAfter configtype.Time `json:"updated_after,omitempty"`
}

type RepositoryDependabotAlerts struct {
	UpdatedAfter configtype.Time `json:"updated_after,omitempty"`
}

type OrgDependabotAlerts struct {
	UpdatedAfter configtype.Time `json:"updated_after,omitempty"`
}

type CopilotOrgMetrics struct {
	Since configtype.Time `json:"since,omitempty"`
}

type CopilotTeamMetrics struct {
	Since configtype.Time `json:"since,omitempty"`
}

type RepositoriesOptions struct {
	UpdatedAfter configtype.Time `json:"updated_after,omitempty"`
}

type RepositoryCommitsOptions struct {
	Until configtype.Time `json:"until,omitempty"`
}

type CodeScanningAlertsOptions struct {
	UpdatedAfter configtype.Time `json:"updated_after,omitempty"`
}

type SecretScanningAlertsOptions struct {
	UpdatedAfter configtype.Time `json:"updated_after,omitempty"`
}

type WorkflowRunsOptions struct {
	// Time to look back for workflow runs in natural date format. Defaults to all workflows.
	// Examples: "14 days ago", "last month"
	CreatedSince    string `json:"created_since,omitempty" jsonschema:"example=14 days ago,example=last month"`
	ParsedTimeSince string `json:"-"`
}

type IssuesOptions struct {
	Milestone string `url:"milestone,omitempty"`

	// State filters issues based on their state. Possible values are: open,
	// closed, all. Default is "open".
	State string `json:"state,omitempty" jsonschema:"default=all"`

	// Assignee filters issues based on their assignee. Possible values are a
	// user name, "none" for issues that are not assigned, "*" for issues with
	// any assigned user.
	Assignee string `json:"assignee,omitempty"`

	// Creator filters issues based on their creator.
	Creator string `json:"creator,omitempty"`

	// Mentioned filters issues to those mentioned a specific user.
	Mentioned string `json:"mentioned,omitempty"`

	// Labels filters issues based on their label.
	Labels []string `json:"labels,omitempty"`

	// Since filters issues by time in natural date format. Defaults to all issues.
	// Examples: "14 days ago", "last month"
	Since string `json:"since,omitempty" jsonschema:"example=14 days ago,example=last month"`

	ParsedSince time.Time `json:"-"`
}

type EnterpriseSettings struct {
	// The base URL of the GitHub Enterprise instance.
	BaseURL string `json:"base_url" jsonschema:"required,minLength=1"`
	// The upload URL of the GitHub Enterprise instance.
	UploadURL string `json:"upload_url" jsonschema:"required,minLength=1"`
}

type AppAuthSpec struct {
	// The GitHub organization to sync from.
	Org string `json:"org" jsonschema:"required,minLength=1"`
	// The GitHub App ID.
	AppID string `json:"app_id" jsonschema:"required,minLength=1"`
	// The path to the private key file used to authenticate the GitHub App.
	PrivateKeyPath string `json:"private_key_path" jsonschema:"minLength=1"`
	// The private key used to authenticate the GitHub App.
	PrivateKey string `json:"private_key" jsonschema:"minLength=1"`
	// The GitHub App installation ID.
	InstallationID string `json:"installation_id" jsonschema:"required,minLength=1"`
}

func (s *Spec) SetDefaults() {
	if s.Concurrency == 0 {
		s.Concurrency = 1500
	}
	if s.DiscoveryConcurrency == 0 {
		s.DiscoveryConcurrency = 1
	}
	if s.Scheduler == nil {
		defaultStrategy := scheduler.StrategyDFS
		s.Scheduler = &defaultStrategy
	}
}

func (s *Spec) Validate() error {
	if s.AccessToken == "" && len(s.AppAuth) == 0 {
		return errors.New("missing personal access token or app auth in configuration")
	}
	if s.EnterpriseSettings != nil {
		if err := s.ValidateEnterpriseSettings(); err != nil {
			return err
		}
	}
	for _, appAuth := range s.AppAuth {
		if appAuth.Org == "" {
			return errors.New("missing org in app auth configuration")
		}
		if appAuth.AppID != "" && (appAuth.PrivateKeyPath == "" && appAuth.PrivateKey == "") {
			return errors.New("missing private key specification in configuration. Please specify it using either `private_key` or `private_key_path`")
		}
		if appAuth.AppID != "" && (appAuth.PrivateKeyPath != "" && appAuth.PrivateKey != "") {
			return errors.New("both private key and private key path specified in configuration. Please remove the configuration for either `private_key_path` or `private_key`")
		}
		if appAuth.AppID != "" && appAuth.InstallationID == "" {
			return errors.New("missing installation id in configuration")
		}
	}
	if len(s.Orgs) == 0 && len(s.Repos) == 0 {
		return errors.New("missing orgs or repos in configuration")
	}
	for _, repo := range s.Repos {
		if err := validateRepo(repo); err != nil {
			return err
		}
	}
	if s.LocalCachePath != "" {
		fileInfo, err := os.Stat(s.LocalCachePath)
		if err != nil && !os.IsNotExist(err) {
			return fmt.Errorf("error accessing local cache path: %w", err)
		}
		if fileInfo != nil && !fileInfo.IsDir() {
			return errors.New("local cache path is not a directory")
		}
	}
	if s.TableOptions.WorkflowRuns.CreatedSince != "" {
		parsedTimeSince, err := naturaldate.Parse(s.TableOptions.WorkflowRuns.CreatedSince, time.Now(), naturaldate.WithDirection(naturaldate.Past))
		if err != nil {
			return fmt.Errorf("failed to parse created_since: %w", err)
		}
		s.TableOptions.WorkflowRuns.ParsedTimeSince = parsedTimeSince.Format(time.RFC3339)
	}

	if s.TableOptions.Issues.Since != "" {
		parsedSince, err := naturaldate.Parse(s.TableOptions.Issues.Since, time.Now(), naturaldate.WithDirection(naturaldate.Past))
		if err != nil {
			return fmt.Errorf("failed to parse since: %w", err)
		}
		s.TableOptions.Issues.ParsedSince = parsedSince
	}
	if s.TableOptions.Issues.State == "" {
		s.TableOptions.Issues.State = "all"
	}

	return nil
}

func (s *Spec) ValidateEnterpriseSettings() error {
	if s.EnterpriseSettings.BaseURL == "" {
		return errors.New("enterprise base url is empty")
	}

	if s.EnterpriseSettings.UploadURL == "" {
		return errors.New("enterprise upload url is empty")
	}

	return nil
}

func validateRepo(repo string) error {
	if repo == "" {
		return errors.New("missing repository")
	}
	if len(splitRepo(repo)) != 2 {
		return fmt.Errorf("invalid repository: %s (should be in <org>/<repo> format)", repo)
	}
	return nil
}

func (Spec) JSONSchemaExtend(sc *jsonschema.Schema) {
	sc.AllOf = []*jsonschema.Schema{
		{
			OneOf: []*jsonschema.Schema{
				{Required: []string{"access_token"}},
				{Required: []string{"app_auth"}},
			},
		},
		{
			AnyOf: []*jsonschema.Schema{
				{Required: []string{"repos"}},
				{Required: []string{"orgs"}},
			},
		},
	}
}

func (AppAuthSpec) JSONSchemaExtend(sc *jsonschema.Schema) {
	sc.OneOf = []*jsonschema.Schema{
		{Required: []string{"private_key_path"}},
		{Required: []string{"private_key"}},
	}
}

//go:embed schema.json
var JSONSchema string
