// Copyright CloudQuery Authors
// This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0.
// If a copy of the MPL was not distributed with this file, You can obtain one at https://mozilla.org/MPL/2.0/.

package client

import (
	"context"
	"os"
	"testing"
	"time"

	"github.com/cloudquery/cloudquery/plugins/source/github/client/services"
	"github.com/cloudquery/plugin-sdk/v4/faker"
	"github.com/cloudquery/plugin-sdk/v4/plugin"
	"github.com/cloudquery/plugin-sdk/v4/scheduler"
	"github.com/cloudquery/plugin-sdk/v4/schema"
	"github.com/cloudquery/plugin-sdk/v4/state"
	"github.com/cloudquery/plugin-sdk/v4/transformers"
	"github.com/golang/mock/gomock"
	"github.com/google/go-github/v69/github"
	"github.com/rs/zerolog"
	"github.com/stretchr/testify/require"
)

type TestOptions struct {
	GraphQLClient GraphQLClient
	Backend       state.Client
}

func GithubMockTestHelper(t *testing.T, table *schema.Table, builder func(*testing.T, *gomock.Controller) services.Github, opts TestOptions) {
	table.IgnoreInTests = false
	t.Helper()
	ctrl := gomock.NewController(t)
	l := zerolog.New(zerolog.NewTestWriter(t)).Output(
		zerolog.ConsoleWriter{Out: os.Stderr, TimeFormat: time.StampMicro},
	).Level(zerolog.DebugLevel).With().Timestamp().Logger()
	sched := scheduler.NewScheduler(scheduler.WithLogger(l))

	var cs github.Repository
	require.NoError(t, faker.FakeObject(&cs))

	someId := int64(5555555)
	cs.Parent = &github.Repository{ID: &someId}
	cs.TemplateRepository = &github.Repository{ID: &someId}
	cs.Source = &github.Repository{ID: &someId}

	b := builder(t, ctrl)

	c := &Client{
		logger: l,
		Github: b,
		orgServices: map[string]services.Github{
			"":        b,
			"testorg": b,
		},
		orgs:                    []string{"testorg"},
		orgRepositories:         map[string][]*github.Repository{"testorg": {&cs}},
		ReposDiscoveryStartTime: time.Now(),
		Backend:                 &state.NoOpClient{},
	}

	if opts.GraphQLClient != nil {
		gql := opts.GraphQLClient
		c.GraphQL = gql
		c.graphqlServices = map[string]GraphQLClient{
			"":        gql,
			"testorg": gql,
		}
	}

	if opts.Backend != nil {
		c.Backend = opts.Backend
	}

	tables := schema.Tables{table}
	if err := transformers.TransformTables(tables); err != nil {
		t.Fatal(err)
	}
	plugin.ValidateSensitiveColumns(t, tables)
	messages, err := sched.SyncAll(context.Background(), c, tables)
	if err != nil {
		t.Fatalf("failed to sync: %v", err)
	}
	plugin.ValidateNoEmptyColumns(t, tables, messages)
}
