// Copyright CloudQuery Authors
// This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0.
// If a copy of the MPL was not distributed with this file, You can obtain one at https://mozilla.org/MPL/2.0/.

package issues

import (
	"context"
	"strings"
	"time"

	"github.com/apache/arrow-go/v18/arrow"
	"github.com/samber/lo"

	"github.com/cloudquery/cloudquery/plugins/source/github/client"
	"github.com/cloudquery/plugin-sdk/v4/schema"
	"github.com/cloudquery/plugin-sdk/v4/transformers"
	"github.com/google/go-github/v69/github"
)

func Issues() *schema.Table {
	return &schema.Table{
		Name:          "github_issues",
		Resolver:      fetchIssues,
		Multiplex:     client.OrgRepositoryMultiplex,
		Transform:     client.TransformWithStruct(&github.Issue{}, transformers.WithPrimaryKeys("ID"), transformers.WithSkipFields("Repository")),
		IsIncremental: true,
		Description: `Lists all issues in a repository, both open and closed.
Pull requests are also included in the list of issues, and can be distinguished by the value in the is_pull_request column. 
This table can be used without the associated permissions if only public resources are requested.

https://docs.github.com/rest/issues/issues#list-repository-issues`,
		PermissionsNeeded: []string{
			"\"Issues\" repository permissions (read)",
		},
		Columns: []schema.Column{
			client.OrgColumn,
			client.RepositoryIDColumn,
			{
				Name: "is_pull_request",
				Type: arrow.FixedWidthTypes.Boolean,
				Resolver: func(_ context.Context, _ schema.ClientMeta, r *schema.Resource, c schema.Column) error {
					isPullRequest := r.Item.(*github.Issue).IsPullRequest()
					return r.Set(c.Name, isPullRequest)
				},
			},
			{
				Name:           "updated_at",
				Type:           arrow.FixedWidthTypes.Timestamp_us,
				Resolver:       schema.PathResolver("UpdatedAt"),
				IncrementalKey: true,
			},
		},
		Relations: schema.Tables{
			pullReviews(),
			timelineEvents(),
		},
	}
}

func fetchIssues(ctx context.Context, meta schema.ClientMeta, _ *schema.Resource, res chan<- any) error {
	c := meta.(*client.Client)
	opts := &github.IssueListByRepoOptions{
		State:       c.Spec.TableOptions.Issues.State,
		Milestone:   c.Spec.TableOptions.Issues.Milestone,
		Assignee:    c.Spec.TableOptions.Issues.Assignee,
		Creator:     c.Spec.TableOptions.Issues.Creator,
		Mentioned:   c.Spec.TableOptions.Issues.Mentioned,
		Labels:      c.Spec.TableOptions.Issues.Labels,
		Since:       c.Spec.TableOptions.Issues.ParsedSince,
		Sort:        "updated",
		Direction:   "asc",
		ListOptions: github.ListOptions{PerPage: 100},
	}

	cursorKey := c.ID() + ` cursor: issues`
	if c.IsIncrementalSync() {
		cursor, err := c.Backend.GetKey(ctx, cursorKey)
		if err != nil {
			return err
		}
		if cursor != "" {
			t, err := time.Parse(time.RFC3339Nano, cursor)
			if err != nil {
				return err
			}
			opts.Since = t
		}
	}

	seenIssues := make(map[int64]bool)
	for {
		lastUpdatedAt, err := fetchIssuesForOpts(ctx, meta, opts, cursorKey, seenIssues, res)
		if err != nil {
			return err
		}
		if lastUpdatedAt == "" {
			break
		}
		t, err := time.Parse(time.RFC3339Nano, lastUpdatedAt)
		if err != nil {
			return err
		}
		opts.Since = t
		opts.Page = 0
	}
	return nil
}

func fetchIssuesForOpts(ctx context.Context, meta schema.ClientMeta, opts *github.IssueListByRepoOptions, cursorKey string, seenIssues map[int64]bool, res chan<- any) (string, error) {
	c := meta.(*client.Client)
	lastUpdatedAt := ""
	const paginationError = "422 Pagination with the page parameter is not supported for large datasets"
	logger := c.Logger(ctx)
	for {
		issues, resp, err := c.Github.Issues.ListByRepo(ctx, c.Org, *c.Repository.Name, opts)
		if err != nil {
			if !strings.Contains(err.Error(), paginationError) {
				return "", err
			}
			logger.Warn().Msg("reached limit of regular pagination for issues, attempting another list operation with updated since parameter")
			return lastUpdatedAt, nil
		}
		res <- dedupIssues(issues, seenIssues)

		if len(issues) > 0 {
			lastUpdatedAt = issues[len(issues)-1].UpdatedAt.Time.Format(time.RFC3339Nano)
			markSeenIssues(issues, seenIssues)
		}
		if c.IsIncrementalSync() && len(issues) > 0 {
			if err = c.Backend.SetKey(ctx, cursorKey, lastUpdatedAt); err != nil {
				return "", err
			}
			if err = c.Backend.Flush(ctx); err != nil {
				return "", err
			}
		}

		if resp.NextPage == 0 {
			break
		}
		opts.Page = resp.NextPage
	}
	return "", nil
}

func markSeenIssues(issues []*github.Issue, seenIssues map[int64]bool) {
	if len(issues) == 0 {
		return
	}
	lastUpdatedTimestamp := issues[len(issues)-1].UpdatedAt.Time
	seenIssues[issues[len(issues)-1].GetID()] = true
	for i := len(issues) - 2; i >= 0; i-- {
		if !issues[i].UpdatedAt.Time.Equal(lastUpdatedTimestamp) {
			break
		}
		seenIssues[issues[i].GetID()] = true
	}
}
func dedupIssues(issues []*github.Issue, seenIssues map[int64]bool) []*github.Issue {
	return lo.Filter(issues, func(issue *github.Issue, _ int) bool {
		_, ok := seenIssues[issue.GetID()]
		return !ok
	})
}
