// Copyright CloudQuery Authors
// This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0.
// If a copy of the MPL was not distributed with this file, You can obtain one at https://mozilla.org/MPL/2.0/.

package organizations

import (
	"context"
	"fmt"
	"time"

	"github.com/apache/arrow-go/v18/arrow"
	"github.com/cloudquery/cloudquery/plugins/source/github/client"
	"github.com/cloudquery/plugin-sdk/v4/schema"
	"github.com/cloudquery/plugin-sdk/v4/transformers"
	"github.com/google/go-github/v69/github"
	"github.com/samber/lo"
)

func alerts() *schema.Table {
	return &schema.Table{
		Name:      "github_organization_dependabot_alerts",
		Resolver:  fetchAlerts,
		Transform: client.TransformWithStruct(&github.DependabotAlert{}, transformers.WithPrimaryKeys("HTMLURL")),
		Columns: []schema.Column{
			client.OrgColumn,
			{
				Name:           "updated_at",
				Type:           arrow.FixedWidthTypes.Timestamp_us,
				Resolver:       schema.PathResolver("UpdatedAt"),
				IncrementalKey: true,
			},
		},
		IsIncremental: true,
	}
}

func fetchAlerts(ctx context.Context, meta schema.ClientMeta, _ *schema.Resource, res chan<- any) error {
	c := meta.(*client.Client)
	opts := &github.ListAlertsOptions{
		ListCursorOptions: github.ListCursorOptions{PerPage: 100},
		Sort:              lo.ToPtr("updated"),
		Direction:         lo.ToPtr("desc"),
	}

	stateKey := fmt.Sprintf("github_organization_dependabot_alerts__updated_at__%s__state", c.ID())
	updatedAtState := time.Time{}

	stateVal, err := c.Backend.GetKey(ctx, stateKey)
	if err != nil {
		return err
	}

	if stateVal != "" {
		updatedAtState, err = time.Parse(time.RFC3339, stateVal)
		if err != nil {
			return err
		}
	}

	now := time.Now()

	if !c.Spec.TableOptions.OrgDependabotAlerts.UpdatedAfter.IsZero() && c.Spec.TableOptions.OrgDependabotAlerts.UpdatedAfter.AsTime(now).After(updatedAtState) {
		updatedAtState = c.Spec.TableOptions.OrgDependabotAlerts.UpdatedAfter.AsTime(now)
	}

	for {
		alerts, resp, err := c.Github.Dependabot.ListOrgAlerts(ctx, c.Org, opts)
		if err != nil {
			return err
		}
		res <- alerts
		if !updatedAtState.IsZero() && len(alerts) > 0 && !alerts[len(alerts)-1].UpdatedAt.After(updatedAtState) {
			break
		}
		if resp.NextPage == 0 {
			break
		}
		opts.ListOptions.Page = resp.NextPage
	}

	return c.Backend.SetKey(ctx, stateKey, now.Format(time.RFC3339))
}
