// Copyright CloudQuery Authors
// This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0.
// If a copy of the MPL was not distributed with this file, You can obtain one at https://mozilla.org/MPL/2.0/.

package repositories

import (
	"context"
	"fmt"
	"time"

	"github.com/apache/arrow-go/v18/arrow"
	"github.com/cloudquery/cloudquery/plugins/source/github/client"
	"github.com/cloudquery/plugin-sdk/v4/schema"
	"github.com/cloudquery/plugin-sdk/v4/transformers"
	"github.com/google/go-github/v69/github"
	"github.com/samber/lo"
)

func alerts() *schema.Table {
	return &schema.Table{
		Name:      "github_repository_dependabot_alerts",
		Resolver:  fetchAlerts,
		Transform: client.TransformWithStruct(&github.DependabotAlert{}, transformers.WithPrimaryKeys("Number")),
		Columns: []schema.Column{
			client.OrgColumn,
			client.RepositoryIDColumn,
			{
				Name:           "updated_at",
				Type:           arrow.FixedWidthTypes.Timestamp_us,
				Resolver:       schema.PathResolver("UpdatedAt"),
				IncrementalKey: true,
			},
		},
		IsIncremental: true,
	}
}

func fetchAlerts(ctx context.Context, meta schema.ClientMeta, parent *schema.Resource, res chan<- any) error {
	c := meta.(*client.Client)
	repo := parent.Item.(*github.Repository)
	opts := &github.ListAlertsOptions{
		ListCursorOptions: github.ListCursorOptions{PerPage: 100},
		Sort:              lo.ToPtr("updated"),
		Direction:         lo.ToPtr("desc"),
	}

	stateKey := fmt.Sprintf("github_repository_dependabot_alerts__updated_at__%s__state", c.ID())
	updatedAtState := time.Time{}

	stateVal, err := c.Backend.GetKey(ctx, stateKey)
	if err != nil {
		return err
	}

	if stateVal != "" {
		updatedAtState, err = time.Parse(time.RFC3339, stateVal)
		if err != nil {
			return err
		}
	}

	now := time.Now()

	if !c.Spec.TableOptions.RepositoryDependabotAlerts.UpdatedAfter.IsZero() && c.Spec.TableOptions.RepositoryDependabotAlerts.UpdatedAfter.AsTime(now).After(updatedAtState) {
		updatedAtState = c.Spec.TableOptions.RepositoryDependabotAlerts.UpdatedAfter.AsTime(now)
	}

	for {
		alerts, resp, err := c.Github.Dependabot.ListRepoAlerts(ctx, c.Org, *repo.Name, opts)
		if err != nil {
			return err
		}
		res <- alerts
		if !updatedAtState.IsZero() && len(alerts) > 0 && !alerts[len(alerts)-1].UpdatedAt.After(updatedAtState) {
			break
		}
		if resp.NextPage == 0 {
			break
		}
		opts.ListOptions.Page = resp.NextPage
	}

	return c.Backend.SetKey(ctx, stateKey, now.Format(time.RFC3339))
}
