// Copyright CloudQuery Authors
// This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0.
// If a copy of the MPL was not distributed with this file, You can obtain one at https://mozilla.org/MPL/2.0/.

package repositories

import (
	"context"
	"fmt"
	"time"

	"github.com/apache/arrow-go/v18/arrow"
	"github.com/cloudquery/cloudquery/plugins/source/github/client"
	"github.com/cloudquery/plugin-sdk/v4/schema"
	"github.com/cloudquery/plugin-sdk/v4/transformers"
	"github.com/google/go-github/v69/github"
)

func Repositories() *schema.Table {
	return &schema.Table{
		Name:      "github_repositories",
		Resolver:  fetchRepositories,
		Multiplex: client.OrgRepositoryMultiplex,
		Transform: client.TransformWithStruct(&github.Repository{},
			transformers.WithPrimaryKeys("ID"),
			transformers.WithSkipFields("CustomProperties"),
		),
		Columns: []schema.Column{
			client.OrgColumn,
			{
				Name:           "updated_at",
				Type:           arrow.FixedWidthTypes.Timestamp_us,
				Resolver:       schema.PathResolver("UpdatedAt"),
				IncrementalKey: true,
			},
		},
		Relations: schema.Tables{
			alerts(),
			branches(),
			collaborators(),
			commits(),
			firstCommits(),
			contributors(),
			customProperties(),
			keys(),
			releases(),
			sboms(),
			secrets(),
		},
		IsIncremental: true,
	}
}

func fetchRepositories(ctx context.Context, meta schema.ClientMeta, _ *schema.Resource, res chan<- any) error {
	c := meta.(*client.Client)

	stateKey := fmt.Sprintf("github_repositories__updated_at__%s__state", c.ID())
	updatedAtState := time.Time{}

	stateVal, err := c.Backend.GetKey(ctx, stateKey)
	if err != nil {
		return err
	}

	if stateVal != "" {
		updatedAtState, err = time.Parse(time.RFC3339, stateVal)
		if err != nil {
			return err
		}
	}

	if !c.Spec.TableOptions.Repositories.UpdatedAfter.IsZero() && c.Spec.TableOptions.Repositories.UpdatedAfter.AsTime(c.ReposDiscoveryStartTime).After(updatedAtState) {
		updatedAtState = c.Spec.TableOptions.Repositories.UpdatedAfter.AsTime(c.ReposDiscoveryStartTime)
	}

	if updatedAtState.IsZero() || c.Repository.UpdatedAt.GetTime().After(updatedAtState) {
		// Repositories are synced during init and multiplexed
		res <- c.Repository
	}

	return c.Backend.SetKey(ctx, stateKey, c.ReposDiscoveryStartTime.Format(time.RFC3339))
}
