// Copyright CloudQuery Authors
// This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0.
// If a copy of the MPL was not distributed with this file, You can obtain one at https://mozilla.org/MPL/2.0/.

package repositories

import (
	"testing"

	"github.com/cloudquery/cloudquery/plugins/source/github/client"
	"github.com/cloudquery/cloudquery/plugins/source/github/client/mocks"
	"github.com/cloudquery/cloudquery/plugins/source/github/client/services"
	"github.com/cloudquery/plugin-sdk/v4/faker"
	"github.com/golang/mock/gomock"
	"github.com/google/go-github/v69/github"
	"github.com/stretchr/testify/require"
)

func buildRepositories(t *testing.T, ctrl *gomock.Controller) services.Github {
	mock := mocks.NewMockGithubRepositoriesService(ctrl)
	dependencyGraph := mocks.NewMockGithubDependencyGraphService(ctrl)

	var sbom github.SBOM
	require.NoError(t, faker.FakeObject(&sbom))
	dependencyGraph.EXPECT().GetSBOM(gomock.Any(), "testorg", gomock.Any()).Return(
		&sbom, &github.Response{}, nil)

	var release github.RepositoryRelease
	require.NoError(t, faker.FakeObject(&release))

	mock.EXPECT().ListReleases(gomock.Any(), "testorg", gomock.Any(), gomock.Any()).Return(
		[]*github.RepositoryRelease{&release}, &github.Response{}, nil)

	var releaseAsset github.ReleaseAsset
	require.NoError(t, faker.FakeObject(&releaseAsset))

	mock.EXPECT().ListReleaseAssets(gomock.Any(), "testorg", gomock.Any(), gomock.Any(), gomock.Any()).Return(
		[]*github.ReleaseAsset{&releaseAsset}, &github.Response{}, nil)

	var branch github.Branch
	require.NoError(t, faker.FakeObject(&branch))

	mock.EXPECT().ListBranches(gomock.Any(), "testorg", gomock.Any(), gomock.Any()).Return(
		[]*github.Branch{&branch}, &github.Response{}, nil)

	var protection github.Protection
	require.NoError(t, faker.FakeObject(&protection))

	mock.EXPECT().GetBranchProtection(gomock.Any(), "testorg", gomock.Any(), gomock.Any()).Return(
		&protection, &github.Response{}, nil)

	dependabot := buildDependabot(t, ctrl)

	var key github.Key
	require.NoError(t, faker.FakeObject(&key))

	mock.EXPECT().ListKeys(gomock.Any(), "testorg", gomock.Any(), gomock.Any()).Return(
		[]*github.Key{&key}, &github.Response{}, nil)

	var collaborators []*github.User
	require.NoError(t, faker.FakeObject(&collaborators))
	mock.EXPECT().ListCollaborators(gomock.Any(), "testorg", gomock.Any(), gomock.Any()).Return(collaborators, &github.Response{}, nil)

	var commits []*github.RepositoryCommit
	require.NoError(t, faker.FakeObject(&commits))
	mock.EXPECT().ListCommits(gomock.Any(), "testorg", gomock.Any(), gomock.Any()).Return(commits, &github.Response{}, nil)

	var firstCommits []*github.RepositoryCommit
	require.NoError(t, faker.FakeObject(&firstCommits))
	mock.EXPECT().ListCommits(gomock.Any(), "testorg", gomock.Any(), gomock.Any()).Return(firstCommits, &github.Response{}, nil)

	var contributors []*github.Contributor
	require.NoError(t, faker.FakeObject(&contributors))
	mock.EXPECT().ListContributors(gomock.Any(), "testorg", gomock.Any(), gomock.Any()).Return(contributors, &github.Response{}, nil)

	var customProps []*github.CustomPropertyValue
	require.NoError(t, faker.FakeObject(&customProps))
	customProps[0].Value = []string{"test", "test"}
	mock.EXPECT().GetAllCustomPropertyValues(gomock.Any(), "testorg", gomock.Any()).Return(customProps, &github.Response{}, nil)

	var commit github.RepositoryCommit
	require.NoError(t, faker.FakeObject(&commit))
	mock.EXPECT().GetCommit(gomock.Any(), "testorg", gomock.Any(), gomock.Any(), gomock.Any()).Return(&commit, &github.Response{}, nil)

	return services.Github{
		Dependabot:      dependabot,
		Repositories:    mock,
		DependencyGraph: dependencyGraph,
	}
}

func TestRepos(t *testing.T) {
	client.GithubMockTestHelper(t, Repositories(), buildRepositories, client.TestOptions{})
}
