// Copyright CloudQuery Authors
// This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0.
// If a copy of the MPL was not distributed with this file, You can obtain one at https://mozilla.org/MPL/2.0/.

package teams

import (
	"testing"

	"github.com/cloudquery/cloudquery/plugins/source/github/client"
	"github.com/cloudquery/cloudquery/plugins/source/github/client/mocks"
	"github.com/cloudquery/cloudquery/plugins/source/github/client/services"
	"github.com/cloudquery/plugin-sdk/v4/faker"
	"github.com/golang/mock/gomock"
	"github.com/google/go-github/v69/github"
	"github.com/stretchr/testify/require"
)

func buildTeams(t *testing.T, ctrl *gomock.Controller) services.Github {
	teamsMock := mocks.NewMockGithubTeamsService(ctrl)

	var cs github.Team
	require.NoError(t, faker.FakeObject(&cs))
	someId := int64(5555555)
	someURL := "https://github.com/orgs/1/teams/test"
	someSlug := "team-slug"
	cs.MembersURL = &someURL
	cs.Parent = &github.Team{ID: &someId}
	cs.Slug = &someSlug

	teamsMock.EXPECT().ListTeams(gomock.Any(), "testorg", gomock.Any()).Return(
		[]*github.Team{&cs}, &github.Response{}, nil)

	var u github.User
	require.NoError(t, faker.FakeObject(&u))
	teamsMock.EXPECT().ListTeamMembersByID(gomock.Any(), gomock.Any(), gomock.Any(), gomock.Any()).Return(
		[]*github.User{&u}, &github.Response{}, nil)

	var r github.Repository
	require.NoError(t, faker.FakeObject(&r))
	r.Parent = &github.Repository{ID: &someId}
	r.TemplateRepository = &github.Repository{ID: &someId}
	r.Source = &github.Repository{ID: &someId}
	r.CustomProperties = map[string]any{"key": "value"}

	teamsMock.EXPECT().ListTeamReposByID(gomock.Any(), gomock.Any(), gomock.Any(), gomock.Any()).Return(
		[]*github.Repository{&r}, &github.Response{}, nil)

	var m github.Membership
	require.NoError(t, faker.FakeObject(&m))

	teamsMock.EXPECT().GetTeamMembershipBySlug(gomock.Any(), "testorg", *cs.Slug, *u.Login).Return(
		&m, &github.Response{}, nil)

	copilotMock := mocks.NewMockGithubCopilotService(ctrl)

	var metrics github.CopilotMetrics
	require.NoError(t, faker.FakeObject(&metrics))

	copilotMock.EXPECT().GetOrganizationTeamMetrics(gomock.Any(), "testorg", someSlug, gomock.Any()).Return([]*github.CopilotMetrics{&metrics}, &github.Response{NextPage: 1}, nil)
	copilotMock.EXPECT().GetOrganizationTeamMetrics(gomock.Any(), "testorg", someSlug, gomock.Any()).Return([]*github.CopilotMetrics{&metrics}, &github.Response{}, nil)

	return services.Github{Teams: teamsMock, Copilot: copilotMock}
}

func TestTeams(t *testing.T) {
	client.GithubMockTestHelper(t, Teams(), buildTeams, client.TestOptions{})
}
