// Copyright CloudQuery Authors
// This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0.
// If a copy of the MPL was not distributed with this file, You can obtain one at https://mozilla.org/MPL/2.0/.

package client

import (
	"crypto/sha256"
	"slices"
	"time"

	"golang.org/x/exp/maps"
	analyticsdata "google.golang.org/api/analyticsdata/v1beta"
)

type row struct {
	Date          time.Time
	Dimensions    map[string]string
	DimensionHash []byte
	Metrics       map[string]string
	DataLoss      bool
}

func convertRows(resp *analyticsdata.RunReportResponse, date time.Time) []*row {
	res := make([]*row, len(resp.Rows))

	for idx, r := range resp.Rows {
		mx := make(map[string]string, len(resp.MetricHeaders))
		for i, mHeader := range resp.MetricHeaders {
			mx[mHeader.Name] = r.MetricValues[i].Value
		}

		dim := make(map[string]string, len(resp.DimensionHeaders))
		for i, dHeader := range resp.DimensionHeaders {
			dim[dHeader.Name] = r.DimensionValues[i].Value
		}

		res[idx] = &row{
			Date:          date,
			Dimensions:    dim,
			DimensionHash: calcMapHash(dim),
			Metrics:       mx,
			DataLoss:      resp.Metadata.DataLossFromOtherRow,
		}
	}

	return res
}

func calcMapHash(m map[string]string) []byte {
	h := sha256.New()
	keys := maps.Keys(m)
	slices.Sort(keys)
	for _, k := range keys {
		h.Write([]byte(k))
		h.Write([]byte(m[k]))
	}

	return h.Sum(nil)
}
