// Copyright CloudQuery Authors
// This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0.
// If a copy of the MPL was not distributed with this file, You can obtain one at https://mozilla.org/MPL/2.0/.

package crm

import (
	"context"

	"github.com/clarkmcc/go-hubspot"
	"github.com/clarkmcc/go-hubspot/generated/v3/deals"
	"github.com/cloudquery/cloudquery/plugins/source/hubspot/client"
	"github.com/cloudquery/plugin-sdk/v4/schema"
)

func fetchDeals(ctx context.Context, meta schema.ClientMeta, parent *schema.Resource, res chan<- any) error {
	cqClient := meta.(*client.Client)

	if cqClient.IsIncrementalSync() {
		return syncIncrementally(ctx, cqClient, res, "deals", "hs_lastmodifieddate")
	}

	return syncAllDeals(ctx, cqClient, res)
}

func syncAllDeals(ctx context.Context, cqClient *client.Client, res chan<- any) error {
	cfg := deals.NewConfiguration()
	cfg.HTTPClient = retryableHTTPClient(cqClient.Logger)

	hubspotClient := deals.NewAPIClient(cfg)

	var after string
	for {
		if err := cqClient.BasicApiRateLimiter.Wait(ctx); err != nil {
			return nil
		}

		req := hubspotClient.BasicApi.
			GetPage(hubspot.WithAuthorizer(ctx, cqClient.Authorizer)).
			Properties(cqClient.Spec.TableOptions.ForTable("hubspot_crm_deals").GetProperties()).
			Associations(cqClient.Spec.TableOptions.ForTable("hubspot_crm_deals").GetAssociations()).
			Limit(maxPageSize)

		if len(after) > 0 {
			req = req.After(after)
		}

		out, resp, err := req.Execute()
		if err != nil {
			return err
		}

		_ = resp.Body.Close()
		res <- out.Results

		if !out.HasPaging() {
			break
		}
		paging := out.GetPaging()
		if !paging.HasNext() {
			break
		}
		next := paging.GetNext()
		after = next.After
		if after == "" {
			break
		}
	}

	return nil
}
