// Copyright CloudQuery Authors
// This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0.
// If a copy of the MPL was not distributed with this file, You can obtain one at https://mozilla.org/MPL/2.0/.

package client

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"net/http"
	"os"
	"reflect"
	"slices"
	"strings"
	"testing"
	"time"

	"github.com/apache/arrow-go/v18/arrow"
	"github.com/cloudquery/plugin-sdk/v4/message"
	"github.com/cloudquery/plugin-sdk/v4/state"
	"github.com/samber/lo"

	"github.com/PagerDuty/go-pagerduty"
	"github.com/cloudquery/plugin-sdk/v4/plugin"
	"github.com/cloudquery/plugin-sdk/v4/scheduler"
	"github.com/cloudquery/plugin-sdk/v4/schema"
	"github.com/cloudquery/plugin-sdk/v4/transformers"
	"github.com/rs/zerolog"
)

type MockHttpClient struct {
	// A map from request path to response object()
	// e.g. "/users" -> []User
	mockResponses map[string]any

	RequestHandler func(req *http.Request) any
}

func (mockHttpClient *MockHttpClient) AddMockResponse(url string, object any) {
	if mockHttpClient.mockResponses == nil {
		mockHttpClient.mockResponses = make(map[string]any)
	}

	mockHttpClient.mockResponses[url] = object
}

type TestOptions struct {
	Spec                  *Spec
	SkipColumnsValidation bool
	SkipRelations         bool
	StateClient           state.Client
}

func PagerdutyMockTestHelper(t *testing.T, table *schema.Table, buildMockHttpClient func() *MockHttpClient, opts TestOptions) []arrow.Record {
	t.Helper()
	l := zerolog.New(zerolog.NewTestWriter(t)).Output(
		zerolog.ConsoleWriter{Out: os.Stderr, TimeFormat: time.StampMicro},
	).Level(zerolog.DebugLevel).With().Timestamp().Logger()
	spec := opts.Spec
	if spec == nil {
		spec = &Spec{
			TeamIds:     nil,
			Concurrency: 0,
		}
	}
	spec.MaxRequestsPerSecond = lo.ToPtr(-1)
	spec.RetryMax = lo.ToPtr(0)

	pagerdutyClient := pagerduty.NewClient("test_auth_token")
	pagerdutyClient.HTTPClient = buildMockHttpClient()

	stateClient := opts.StateClient
	if stateClient == nil {
		stateClient = &state.NoOpClient{}
	}

	c, err := New(l, *spec, stateClient, WithClient(pagerdutyClient))
	if err != nil {
		t.Fatal(err)
	}

	tables := schema.Tables{table}
	if err := transformers.TransformTables(tables); err != nil {
		t.Fatal(err)
	}
	if opts.SkipRelations {
		tables[0].Relations = nil
	}
	plugin.ValidateSensitiveColumns(t, tables)
	sched := scheduler.NewScheduler(scheduler.WithLogger(l))
	messages, err := sched.SyncAll(context.Background(), c, tables)
	if err != nil {
		t.Fatalf("failed to sync: %v", err)
	}
	if !opts.SkipColumnsValidation {
		plugin.ValidateNoEmptyColumns(t, tables, messages)
	}
	gotRecords := []arrow.Record{}
	for _, msg := range messages {
		if m, ok := msg.(*message.SyncInsert); ok {
			// flatten the record so it's easier to compare
			for row := range m.Record.NumRows() {
				currentRow := m.Record.NewSlice(row, row+1)
				gotRecords = append(gotRecords, currentRow)
			}
		}
	}
	return gotRecords
}

func (mockHttpClient *MockHttpClient) getResponse(req *http.Request) any {
	if mockHttpClient.RequestHandler != nil {
		return mockHttpClient.RequestHandler(req)
	}
	mockResponseObject := mockHttpClient.mockResponses[strings.TrimRight(req.URL.Path, "/")]
	return mockResponseObject
}

func (mockHttpClient *MockHttpClient) Do(req *http.Request) (*http.Response, error) {
	mockResponseObject := mockHttpClient.getResponse(req)
	marshaledMockResponse, err := json.Marshal(mockResponseObject)

	if err != nil {
		panic(err)
	}

	httpResponse := http.Response{
		Proto:      "HTTP/1.1",
		ProtoMajor: 1,
		ProtoMinor: 1,
		Header:     http.Header{},
	}
	if errorResponseObject, ok := mockResponseObject.(pagerduty.APIError); ok {
		httpResponse.Status = fmt.Sprintf("%d %s", errorResponseObject.StatusCode, http.StatusText(errorResponseObject.StatusCode))
		httpResponse.StatusCode = errorResponseObject.StatusCode
		httpResponse.Header.Set("Content-Type", "application/json")
		type pderr struct {
			ErrorObject pagerduty.APIErrorObject `json:"error"`
		}
		res, _ := json.Marshal(pderr{ErrorObject: errorResponseObject.APIError.ErrorObject})
		httpResponse.Body = io.NopCloser(bytes.NewReader(res))
	} else {
		httpResponse.Status = "200 OK"
		httpResponse.StatusCode = 200
		httpResponse.Body = io.NopCloser(bytes.NewReader(marshaledMockResponse))
	}

	return &httpResponse, nil
}

// Timestamp fields such as `created_at` arrive as `string`s from the API.
// this function pusts valid `RFC3339` timestamps into fields like `DeletedAt`, `CreatedAt`...
// Mostly copy-paste from the `plugin-sdk` faker.
// Receives an interface that is a pointer to a struct, and only looks at fields one level deep.
// Pionter-to-pointer structs are supported.
func FakeStringTimestamps(ptrObj any) error {
	timestampFieldNames := []string{
		"CreateAt", "CreatedAt", "DeletedAt", "LastStatusChangeAt", "StartTime", "EndTime", "LastIncidentTimestamp",
		"UpdatedAt", "ResolvedAt", "Start", "End",
	}

	ptrType := reflect.TypeOf(ptrObj) // reflection-type is a pointer
	ptrKind := ptrType.Kind()
	ptrValue := reflect.ValueOf(ptrObj) // a reflection-value that is a pointer

	if ptrKind != reflect.Ptr {
		return errors.New("object must be a pointer")
	}

	if ptrValue.IsNil() {
		return errors.New("object must not be nil")
	}

	if ptrValue.Elem().Kind() == reflect.Ptr {
		return FakeStringTimestamps(ptrValue.Elem().Interface())
	}

	if ptrValue.Elem().Kind() != reflect.Struct {
		return errors.New("object must be a pointer to a struct")
	}

	structValue := ptrValue.Elem()
	structType := structValue.Type()

	for i := 0; i < structType.NumField(); i++ {
		field := structType.Field(i)
		fieldType := field.Type
		fieldKind := fieldType.Kind()
		fieldValue := structValue.Field(i)

		if fieldKind == reflect.String {
			if !fieldValue.CanSet() { // don't panic on unexported fields
				continue
			}

			if slices.Contains(timestampFieldNames, field.Name) {
				fieldValue.Set(reflect.ValueOf(time.Now().Format(time.RFC3339)))
			}
		}
	}

	return nil
}
