// Copyright CloudQuery Authors
// This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0.
// If a copy of the MPL was not distributed with this file, You can obtain one at https://mozilla.org/MPL/2.0/.

package incidents

import (
	"context"
	"strings"
	"time"

	"github.com/PagerDuty/go-pagerduty"
	"github.com/cloudquery/cloudquery/plugins/source/pagerduty/client"
	"github.com/cloudquery/plugin-sdk/v4/schema"
	"github.com/samber/lo"
)

// the limit is 6 months, but when using 180 days sometimes PD throws an error, so we use 170 to be safe
const maxQueryWindowDays = 170

var (
	resolvedStatusList   = []string{"resolved"}
	unresolvedStatusList = []string{"triggered", "acknowledged"}
)

func fetchIncidents(ctx context.Context, meta schema.ClientMeta, parent *schema.Resource, res chan<- any) error {
	c := meta.(*client.Client)
	if !c.IsIncrementalSync() {
		return syncAllIncidents(ctx, c, res)
	}

	// for incremental syncing we:
	// 		1. syncIncidents all unresolved incidents
	//		2. syncIncidents all resolved incidents that we have not yet seen
	incident, err := syncUnresolvedIncidents(ctx, c, res)
	if err != nil {
		return err
	}

	if err := syncResolvedIncidentsIncrementally(ctx, c, res); err != nil {
		return err
	}

	if incident == nil {
		return nil
	}

	return saveCursor(ctx, c, incident.CreatedAt)
}

func syncAllIncidents(ctx context.Context, cqClient *client.Client, res chan<- any) error {
	// a nil status list means we don't filter by status
	_, err := syncIncidents(ctx, cqClient, res, nil, false)
	return err
}

func syncUnresolvedIncidents(ctx context.Context, cqClient *client.Client, res chan<- any) (*pagerduty.Incident, error) {
	return syncIncidents(ctx, cqClient, res, unresolvedStatusList, false)
}

func syncResolvedIncidentsIncrementally(ctx context.Context, cqClient *client.Client, res chan<- any) error {
	_, err := syncIncidents(ctx, cqClient, res, resolvedStatusList, true)
	if err != nil {
		return err
	}
	return err
}

func syncIncidents(ctx context.Context, cqClient *client.Client, res chan<- any, statuses []string, isResolvedIncrementalSync bool) (*pagerduty.Incident, error) {
	var err error

	upperBound := cqClient.SyncTime
	if !cqClient.Spec.TableOptions.Incidents.Until.IsZero() {
		// The option is validated during plugin initialization
		upperBound = cqClient.Spec.TableOptions.Incidents.Until.AsTime(cqClient.SyncTime)
	}

	// the upper bound is exclusive, so we add a second to make it match the lower bound behaviour
	upperBound = upperBound.Add(time.Second)

	lowerBound := time.Time{}
	if isResolvedIncrementalSync {
		lowerBound, err = readCursor(ctx, cqClient)
		if err != nil {
			return nil, err
		}
	}

	if !cqClient.Spec.TableOptions.Incidents.Since.IsZero() {
		// The option is validated during plugin initialization
		lowerBoundFromSpec := cqClient.Spec.TableOptions.Incidents.Since.AsTime(cqClient.SyncTime)
		if lowerBoundFromSpec.After(lowerBound) {
			lowerBound = lowerBoundFromSpec
		}
	}

	cqClient.Logger(ctx).Debug().
		Any("statuses", statuses).
		Msgf("Detected lower bound '%s' and upper bound '%s'", lowerBound.Format(time.RFC3339), upperBound.Format(time.RFC3339))

	var firstIncident *pagerduty.Incident

	for since := lowerBound; since.Before(upperBound); since = since.AddDate(0, 0, maxQueryWindowDays) {
		var previous []pagerduty.Incident
		var response *pagerduty.ListIncidentsResponse

		for offset := 0; ; {
			if offset+client.MaxPaginationLimit > client.MaxRows {
				offset = 0
				parsedCreatedAt, err := time.Parse(time.RFC3339, response.Incidents[len(response.Incidents)-1].CreatedAt)
				if err != nil {
					return nil, err
				}
				since = parsedCreatedAt
				previous = response.Incidents
			}

			options := pagerduty.ListIncidentsOptions{
				Limit:    client.MaxPaginationLimit,
				Offset:   uint(offset),
				TeamIDs:  cqClient.Spec.TeamIds,
				Statuses: statuses,
				SortBy:   "created_at",
			}

			if since.IsZero() {
				// if there is no specified `since` date we have to detect the date of the first incident by
				// initially querying using DateRange=all and then incrementing that until we reach the `until` date
				options.DateRange = "all"
			} else {
				until := since.AddDate(0, 0, maxQueryWindowDays)
				if until.After(upperBound) {
					until = upperBound
				}
				options.Since = since.UTC().Format(time.RFC3339) // since is inclusive
				options.Until = until.UTC().Format(time.RFC3339) // until is exclusive
			}

			response, err = cqClient.PagerdutyClient.ListIncidentsWithContext(ctx, options)
			if err != nil {
				return nil, err
			}
			if len(response.Incidents) == 0 {
				break
			}

			if firstIncident == nil {
				if since.IsZero() {
					// because we defaulted to DateRange=all we have to manually make sure that all the incidents
					// were created before the `until` date (happens only on the first page of the results)
					response.Incidents = lo.Filter(response.Incidents, func(incident pagerduty.Incident, _ int) bool {
						createdAt, err := time.Parse(time.RFC3339, incident.CreatedAt)
						if err != nil {
							return false
						}
						if since.IsZero() {
							since = createdAt
							cqClient.Logger(ctx).Debug().Any("statuses", statuses).Msgf("detected since=%s", since)
						}
						return createdAt.Before(upperBound)
					})
					if len(response.Incidents) == 0 {
						return nil, nil
					}
				}

				firstIncident = &response.Incidents[0]
			}

			offset += len(response.Incidents)

			if len(previous) > 0 {
				// de-duplicate incidents after the lower bound change because the `since` filtering is inclusive
				response.Incidents = deduplicateIncidents(response.Incidents, previous)
				previous = nil
			}

			res <- response.Incidents
			if !response.More {
				break
			}
		}
	}

	return firstIncident, nil
}

func deduplicateIncidents(incidents []pagerduty.Incident, previous []pagerduty.Incident) []pagerduty.Incident {
	seenIncidents := map[string]any{}
	for _, incident := range previous {
		seenIncidents[incident.ID] = nil
	}
	return lo.Filter(incidents, func(incident pagerduty.Incident, _ int) bool {
		_, ok := seenIncidents[incident.ID]
		return !ok
	})
}

func cursorKey(cqClient *client.Client) string {
	key := cqClient.ID() + "-" + incidentsTableName + "-cursor-v2"
	if len(cqClient.Spec.TeamIds) == 0 {
		return key
	}
	return key + "-teams-" + strings.Join(cqClient.Spec.TeamIds, "-")
}

func saveCursor(ctx context.Context, cqClient *client.Client, createdAt string) error {
	key := cursorKey(cqClient)
	formattedValue, err := time.Parse(time.RFC3339, createdAt)
	if err != nil {
		return err
	}
	cqClient.Logger(ctx).Info().Str("cursor_key", key).Time("state_value", formattedValue).Msg("Saving state")
	if err := cqClient.Backend.SetKey(ctx, key, formattedValue.Format(time.RFC3339)); err != nil {
		return err
	}
	return cqClient.Backend.Flush(ctx)
}

func readCursor(ctx context.Context, cqClient *client.Client) (time.Time, error) {
	key := cursorKey(cqClient)
	value, err := cqClient.Backend.GetKey(ctx, key)
	if err != nil {
		return time.Time{}, err
	}
	if value == "" {
		cqClient.Logger(ctx).Info().Str("cursor_key", key).Msg("No state found")
		return time.Time{}, nil
	}
	parsedTime, err := time.Parse(time.RFC3339, value)
	if err != nil {
		return time.Time{}, err
	}
	cqClient.Logger(ctx).Info().Str("cursor_key", key).Time("state_value", parsedTime).Msg("Read state")
	return parsedTime, nil
}
