// Copyright CloudQuery Authors
// This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0.
// If a copy of the MPL was not distributed with this file, You can obtain one at https://mozilla.org/MPL/2.0/.

package recipes

import "github.com/stripe/stripe-go/v80"

const (
	createdStateParam     = "Created"
	createdStateParamPath = "CreatedRange.GreaterThan"
	createdStateParamType = "int64"
	createdListParams     = `CreatedRange: &stripe.RangeQueryParams{},`
)

var AllResources = []*Resource{
	{
		DataStruct:     &stripe.Account{},
		ListParams:     createdListParams,
		StateParamName: createdStateParam,
		StateParamPath: createdStateParamPath,
		StateParamType: createdStateParamType,
		Children: []*Resource{
			{
				DataStruct: &stripe.Capability{},
				ListParams: `Account: stripe.String(p.ID),`,
			},
		},
	},
	{
		DataStruct:  &stripe.ApplePayDomain{},
		Description: "https://stripe.com/docs/api",
	},
	{
		DataStruct:     &stripe.ApplicationFee{},
		ListParams:     createdListParams,
		StateParamName: createdStateParam,
		StateParamPath: createdStateParamPath,
		StateParamType: createdStateParamType,
		Children: []*Resource{
			{
				DataStruct: &stripe.FeeRefund{},
				ListParams: `ID: stripe.String(p.ID),`,
			},
		},
	},
	{
		DataStruct: &stripe.Balance{},
		Single:     true,
		Service:    "balance",
	},
	{
		DataStruct:     &stripe.BalanceTransaction{},
		Service:        "balance",
		ListParams:     createdListParams,
		StateParamName: createdStateParam,
		StateParamPath: createdStateParamPath,
		StateParamType: createdStateParamType,
	},
	{
		DataStruct:  &stripe.BillingPortalConfiguration{},
		Service:     "billing_portal",
		Description: "https://stripe.com/docs/api/customer_portal/configuration",
	},
	{
		DataStruct:     &stripe.Charge{},
		IgnoreInTests:  []string{"Destination", "Dispute", "Level3", "Source"},
		ListParams:     createdListParams,
		StateParamName: createdStateParam,
		StateParamPath: createdStateParamPath,
		StateParamType: createdStateParamType,
	},
	{
		DataStruct:    &stripe.CheckoutSession{},
		Service:       "checkout",
		Description:   "https://stripe.com/docs/api/checkout/sessions",
		IgnoreInTests: []string{"PaymentMethodConfigurationDetails"},
		ExtraChildren: []string{"CheckoutSessionLineItems()"},
	},
	{
		DataStruct: &stripe.CountrySpec{
			// These fields are not faked correctly, so we set them manually.
			VerificationFields: map[stripe.AccountBusinessType]*stripe.VerificationFieldsList{
				stripe.AccountBusinessTypeCompany: {
					AdditionalFields: []string{"test1"},
					Minimum:          []string{"test2"},
				},
			},
		},
	},
	{
		DataStruct:     &stripe.Coupon{},
		ListParams:     createdListParams,
		StateParamName: createdStateParam,
		StateParamPath: createdStateParamPath,
		StateParamType: createdStateParamType,
	},
	{
		DataStruct: &stripe.CreditNote{},
	},
	{
		DataStruct:     &stripe.Customer{},
		IgnoreInTests:  []string{"DefaultSource"},
		ListParams:     createdListParams,
		StateParamName: createdStateParam,
		StateParamPath: createdStateParamPath,
		StateParamType: createdStateParamType,
	},
	{
		DataStruct:     &stripe.Dispute{},
		ListParams:     createdListParams,
		StateParamName: createdStateParam,
		StateParamPath: createdStateParamPath,
		StateParamType: createdStateParamType,
		IgnoreInTests:  []string{"PaymentMethodDetails"},
	},
	{
		DataStruct:     &stripe.Event{},
		ListParams:     createdListParams,
		StateParamName: createdStateParam,
		StateParamPath: createdStateParamPath,
		StateParamType: createdStateParamType,
	},
	{
		DataStruct:     &stripe.FileLink{},
		ListParams:     createdListParams,
		StateParamName: createdStateParam,
		StateParamPath: createdStateParamPath,
		StateParamType: createdStateParamType,
	},
	{
		DataStruct:     &stripe.File{},
		ListParams:     createdListParams,
		StateParamName: createdStateParam,
		StateParamPath: createdStateParamPath,
		StateParamType: createdStateParamType,
	},
	{
		DataStruct:     &stripe.IdentityVerificationReport{},
		Service:        "identity",
		ListParams:     createdListParams,
		StateParamName: createdStateParam,
		StateParamPath: createdStateParamPath,
		StateParamType: createdStateParamType,
		Description:    "https://stripe.com/docs/api/identity/verification_reports",
	},
	{
		DataStruct:     &stripe.Invoice{},
		IgnoreInTests:  []string{"DefaultSource", "Rendering", "RenderingOptions", "SubscriptionDetails"},
		ListParams:     createdListParams,
		StateParamName: createdStateParam,
		StateParamPath: createdStateParamPath,
		StateParamType: createdStateParamType,
	},
	{
		DataStruct:    &stripe.InvoiceItem{},
		Service:       "invoices",
		Description:   "https://stripe.com/docs/api/invoiceitems",
		IgnoreInTests: []string{"Plan"},
	},
	{
		DataStruct:     &stripe.IssuingAuthorization{},
		Service:        "issuing",
		ListParams:     createdListParams,
		StateParamName: createdStateParam,
		StateParamPath: createdStateParamPath,
		StateParamType: createdStateParamType,
		Description:    "https://stripe.com/docs/api/issuing/authorizations",
		IgnoreInTests:  []string{"Token"},
	},
	{
		DataStruct:     &stripe.IssuingCardholder{},
		Service:        "issuing",
		ListParams:     createdListParams,
		StateParamName: createdStateParam,
		StateParamPath: createdStateParamPath,
		StateParamType: createdStateParamType,
		Description:    "https://stripe.com/docs/api/issuing/cardholders",
		IgnoreInTests:  []string{"PreferredLocales"},
	},
	{
		DataStruct:     &stripe.IssuingCard{},
		Service:        "issuing",
		ListParams:     createdListParams,
		StateParamName: createdStateParam,
		StateParamPath: createdStateParamPath,
		StateParamType: createdStateParamType,
		Description:    "https://stripe.com/docs/api/issuing/cards",
	},
	{
		DataStruct:     &stripe.IssuingDispute{},
		Service:        "issuing",
		ListParams:     createdListParams,
		StateParamName: createdStateParam,
		StateParamPath: createdStateParamPath,
		StateParamType: createdStateParamType,
		Description:    "https://stripe.com/docs/api/issuing/disputes",
	},
	{
		DataStruct:     &stripe.IssuingTransaction{},
		Service:        "issuing",
		ListParams:     createdListParams,
		StateParamName: createdStateParam,
		StateParamPath: createdStateParamPath,
		StateParamType: createdStateParamType,
		Description:    "https://stripe.com/docs/api/issuing/transactions",
		IgnoreInTests:  []string{"Token", "NetworkData"},
	},
	{
		DataStruct:     &stripe.PaymentIntent{},
		Service:        "payment",
		IgnoreInTests:  []string{"Source", "PaymentMethodConfigurationDetails"},
		ListParams:     createdListParams,
		StateParamName: createdStateParam,
		StateParamPath: createdStateParamPath,
		StateParamType: createdStateParamType,
	},
	{
		DataStruct:    &stripe.PaymentLink{},
		Service:       "payment",
		ExtraChildren: []string{"PaymentLinkLineItems()"},
		IgnoreInTests: []string{"Application", "Restrictions"},
	},
	{
		DataStruct: &stripe.PaymentMethod{},
		Service:    "payment",
		IgnoreInTests: []string{
			"Affirm",
			"AfterpayClearpay",
			"Alipay",
			"AmazonPay",
			"Bancontact",
			"BLIK",
			"CardPresent",
			"Cashapp",
			"CustomerBalance",
			"Giropay",
			"Grabpay",
			"InteracPresent",
			"Konbini",
			"Mobilepay",
			"Multibanco",
			"OXXO",
			"PayNow",
			"Paypal",
			"Pix",
			"PromptPay",
			"RevolutPay",
			"Swish",
			"TWINT",
			"WeChatPay",
			"Zip",
		},
	},
	{
		DataStruct:     &stripe.Payout{},
		ListParams:     createdListParams,
		StateParamName: createdStateParam,
		StateParamPath: createdStateParamPath,
		StateParamType: createdStateParamType,
	},
	{
		DataStruct:     &stripe.Plan{},
		ListParams:     createdListParams,
		StateParamName: createdStateParam,
		StateParamPath: createdStateParamPath,
		StateParamType: createdStateParamType,
	},
	{
		DataStruct:     &stripe.Price{},
		ListParams:     createdListParams,
		StateParamName: createdStateParam,
		StateParamPath: createdStateParamPath,
		StateParamType: createdStateParamType,
		ExpandFields:   []string{"data.currency_options", "data.tiers"},
	},
	{
		DataStruct:     &stripe.Product{},
		IgnoreInTests:  []string{"Attributes", "DeactivateOn", "Features"},
		ListParams:     createdListParams,
		StateParamName: createdStateParam,
		StateParamPath: createdStateParamPath,
		StateParamType: createdStateParamType,
	},
	{
		DataStruct:     &stripe.PromotionCode{},
		ListParams:     createdListParams,
		StateParamName: createdStateParam,
		StateParamPath: createdStateParamPath,
		StateParamType: createdStateParamType,
	},
	{
		DataStruct: &stripe.Quote{},
	},
	{
		DataStruct:  &stripe.RadarEarlyFraudWarning{},
		Service:     "radar",
		Description: "https://stripe.com/docs/api/radar/early_fraud_warnings",
	},
	{
		DataStruct:     &stripe.Refund{},
		ListParams:     createdListParams,
		StateParamName: createdStateParam,
		StateParamPath: createdStateParamPath,
		StateParamType: createdStateParamType,
	},
	{
		DataStruct:     &stripe.ReportingReportRun{},
		Service:        "reporting",
		ListParams:     createdListParams,
		StateParamName: createdStateParam,
		StateParamPath: createdStateParamPath,
		StateParamType: createdStateParamType,
		Description:    "https://stripe.com/docs/api/reporting/report_run",
	},
	{
		DataStruct:  &stripe.ReportingReportType{},
		Service:     "reporting",
		Description: "https://stripe.com/docs/api/reporting/report_type",
	},
	{
		DataStruct:     &stripe.Review{},
		ListParams:     createdListParams,
		StateParamName: createdStateParam,
		StateParamPath: createdStateParamPath,
		StateParamType: createdStateParamType,
		Description:    "https://stripe.com/docs/api/radar/reviews",
	},
	{
		DataStruct:     &stripe.ShippingRate{},
		ListParams:     createdListParams,
		StateParamName: createdStateParam,
		StateParamPath: createdStateParamPath,
		StateParamType: createdStateParamType,
	},
	{
		DataStruct:  &stripe.SigmaScheduledQueryRun{},
		Service:     "sigma",
		Description: "https://stripe.com/docs/api/sigma/scheduled_queries",
	},
	{
		DataStruct:     &stripe.Subscription{},
		Service:        "subscription",
		IgnoreInTests:  []string{"DefaultSource"},
		ListParams:     createdListParams,
		StateParamName: createdStateParam,
		StateParamPath: createdStateParamPath,
		StateParamType: createdStateParamType,
	},
	{
		DataStruct:     &stripe.SubscriptionSchedule{},
		Service:        "subscription",
		ListParams:     createdListParams,
		StateParamName: createdStateParam,
		StateParamPath: createdStateParamPath,
		StateParamType: createdStateParamType,
	},
	{
		DataStruct: &stripe.TaxCode{},
		Service:    "tax",
	},
	{
		DataStruct:     &stripe.TaxRate{},
		Service:        "tax",
		ListParams:     createdListParams,
		StateParamName: createdStateParam,
		StateParamPath: createdStateParamPath,
		StateParamType: createdStateParamType,
	},
	{
		DataStruct:    &stripe.TerminalConfiguration{},
		Service:       "terminal",
		Description:   "https://stripe.com/docs/api/terminal/configuration",
		IgnoreInTests: []string{"Offline"},
	},
	{
		DataStruct:  &stripe.TerminalLocation{},
		Service:     "terminal",
		Description: "https://stripe.com/docs/api/terminal/locations",
	},
	{
		DataStruct:  &stripe.TerminalReader{},
		Service:     "terminal",
		Description: "https://stripe.com/docs/api/terminal/readers",
	},
	{
		DataStruct:     &stripe.Topup{},
		IgnoreInTests:  []string{"Source"},
		ListParams:     createdListParams,
		StateParamName: createdStateParam,
		StateParamPath: createdStateParamPath,
		StateParamType: createdStateParamType,
	},
	{
		DataStruct:     &stripe.Transfer{},
		ListParams:     createdListParams,
		StateParamName: createdStateParam,
		StateParamPath: createdStateParamPath,
		StateParamType: createdStateParamType,
	},
	{
		DataStruct:     &stripe.TreasuryFinancialAccount{},
		Service:        "treasury",
		ListParams:     createdListParams,
		StateParamName: createdStateParam,
		StateParamPath: createdStateParamPath,
		StateParamType: createdStateParamType,
		Description:    "https://stripe.com/docs/api/treasury/financial_accounts",
		Children: []*Resource{
			{
				DataStruct:  &stripe.TreasuryCreditReversal{},
				ListParams:  `FinancialAccount: stripe.String(p.ID),`,
				Description: "https://stripe.com/docs/api/treasury/credit_reversals",
			},
			{
				DataStruct:  &stripe.TreasuryDebitReversal{},
				ListParams:  `FinancialAccount: stripe.String(p.ID),`,
				Description: "https://stripe.com/docs/api/treasury/debit_reversals",
			},
			{
				DataStruct:  &stripe.TreasuryInboundTransfer{},
				ListParams:  `FinancialAccount: stripe.String(p.ID),`,
				Description: "https://stripe.com/docs/api/treasury/inbound_transfers",
			},
			{
				DataStruct:  &stripe.TreasuryOutboundPayment{},
				ListParams:  `FinancialAccount: stripe.String(p.ID),`,
				Description: "https://stripe.com/docs/api/treasury/outbound_payments",
			},
			{
				DataStruct:  &stripe.TreasuryOutboundTransfer{},
				ListParams:  `FinancialAccount: stripe.String(p.ID),`,
				Description: "https://stripe.com/docs/api/treasury/outbound_transfers",
			},
			{
				DataStruct:  &stripe.TreasuryReceivedCredit{},
				ListParams:  `FinancialAccount: stripe.String(p.ID),`,
				Description: "https://stripe.com/docs/api/treasury/received_credits",
			},
			{
				DataStruct:  &stripe.TreasuryReceivedDebit{},
				ListParams:  `FinancialAccount: stripe.String(p.ID),`,
				Description: "https://stripe.com/docs/api/treasury/received_debits",
			},
			{
				DataStruct: &stripe.TreasuryTransactionEntry{},
				ListParams: `FinancialAccount: stripe.String(p.ID),
` + createdListParams,
				StateParamName: createdStateParam,
				StateParamPath: createdStateParamPath,
				StateParamType: createdStateParamType,
				Description:    "https://stripe.com/docs/api/treasury/transaction_entries",
			},
			{
				DataStruct: &stripe.TreasuryTransaction{},
				ListParams: `FinancialAccount: stripe.String(p.ID),
` + createdListParams,
				StateParamName: createdStateParam,
				StateParamPath: createdStateParamPath,
				StateParamType: createdStateParamType,
				Description:    "https://stripe.com/docs/api/treasury/transactions",
			},
		},
	},
	{
		DataStruct: &stripe.WebhookEndpoint{},
	},
}
