// Copyright CloudQuery Authors
// This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0.
// If a copy of the MPL was not distributed with this file, You can obtain one at https://mozilla.org/MPL/2.0/.

package main

import (
	"fmt"
	"go/ast"
	"go/parser"
	"go/token"
	"log"
	"os"
	"strings"

	"github.com/bmatcuk/doublestar/v4"
)

func isTableName(str string) bool {
	return strings.Contains(strings.ToLower(str), "table") || str == "name"
}

func trimTableName(str string) string {
	return strings.Trim(str, "\"`")
}

func parseDocsTables() map[string]bool {
	tablesMap := make(map[string]bool)
	tablesReadmes, err := doublestar.Glob(os.DirFS("../../"), "plugins/source/**/docs/tables/README.md", doublestar.WithFailOnPatternNotExist(), doublestar.WithFilesOnly())
	if err != nil {
		panic(err)
	}

	for _, readme := range tablesReadmes {
		if strings.HasPrefix(readme, "plugins/source/test") {
			continue
		}
		content, err := os.ReadFile("../../" + readme)
		if err != nil {
			panic(err)
		}
		lines := strings.Split(string(content), "\n")
		for _, line := range lines {
			pos1 := strings.Index(line, "[")
			if pos1 == -1 {
				continue
			}
			pos2 := strings.Index(line, "]")
			table := line[pos1+1 : pos2]
			tablesMap[table] = true
		}
	}

	return tablesMap
}

func isVarOrConstDecl(decl ast.Decl) bool {
	genDecl, ok := decl.(*ast.GenDecl)
	if !ok {
		return false
	}
	return genDecl.Tok == token.VAR || genDecl.Tok == token.CONST
}

func findTableNamesInVariablesAndConstDeclarations(file *ast.File) []string {
	var tableNames []string
	for _, decl := range file.Decls {
		if !isVarOrConstDecl(decl) {
			continue
		}
		for _, spec := range decl.(*ast.GenDecl).Specs {
			valueSpec, ok := spec.(*ast.ValueSpec)
			if !ok {
				continue
			}
			for _, ident := range valueSpec.Names {
				if isTableName(ident.Name) && len(valueSpec.Values) == 1 {
					basicLit, ok := valueSpec.Values[0].(*ast.BasicLit)
					if !ok {
						continue
					}
					tableName := trimTableName(basicLit.Value)
					tableNames = append(tableNames, tableName)
				}
			}
		}
	}
	return tableNames
}

func getTableFunctionDeclarations(file *ast.File) []*ast.FuncDecl {
	var funcDeclarations []*ast.FuncDecl
	for _, decl := range file.Decls {
		fn, ok := decl.(*ast.FuncDecl)
		if !ok {
			continue
		}
		if fn.Type.Results == nil {
			continue
		}
		if len(fn.Type.Results.List) != 1 {
			continue
		}
		resultType, ok := fn.Type.Results.List[0].Type.(*ast.StarExpr)
		if !ok {
			continue
		}
		xType, ok := resultType.X.(*ast.SelectorExpr)
		if !ok {
			continue
		}
		if xType.X.(*ast.Ident).Name != "schema" {
			continue
		}
		if xType.Sel.Name != "Table" {
			continue
		}
		funcDeclarations = append(funcDeclarations, fn)
	}
	return funcDeclarations
}

func nameFromReturnStmt(stmt *ast.ReturnStmt) string {
	if len(stmt.Results) != 1 {
		return ""
	}
	unaryExpr, ok := stmt.Results[0].(*ast.UnaryExpr)
	if !ok {
		return ""
	}
	compositeLit, ok := unaryExpr.X.(*ast.CompositeLit)
	if !ok {
		return ""
	}
	for _, elt := range compositeLit.Elts {
		kvExpr, ok := elt.(*ast.KeyValueExpr)
		if !ok {
			continue
		}
		ident, ok := kvExpr.Key.(*ast.Ident)
		if !ok || ident.Name != "Name" {
			continue
		}
		basicLit, ok := kvExpr.Value.(*ast.BasicLit)
		if !ok {
			continue
		}
		return trimTableName(basicLit.Value)
	}
	return ""
}

func nameFromAssignStmt(stmt *ast.AssignStmt) string {
	if len(stmt.Lhs) != 1 || len(stmt.Rhs) != 1 {
		return ""
	}
	ident, ok := stmt.Lhs[0].(*ast.Ident)
	if !ok || !isTableName(ident.Name) {
		return ""
	}
	switch rhs := stmt.Rhs[0].(type) {
	case *ast.BasicLit:
		return trimTableName(rhs.Value)
	case *ast.UnaryExpr:
		compositeLit, ok := rhs.X.(*ast.CompositeLit)
		if !ok {
			return ""
		}
		for _, elt := range compositeLit.Elts {
			kvExpr, ok := elt.(*ast.KeyValueExpr)
			if !ok {
				continue
			}
			ident, ok := kvExpr.Key.(*ast.Ident)
			if !ok || ident.Name != "Name" {
				continue
			}
			basicLit, ok := kvExpr.Value.(*ast.BasicLit)
			if !ok {
				continue
			}
			return trimTableName(basicLit.Value)
		}
	}
	return ""
}

func findTableNamesInFunctionsDeclarations(funcDeclarations []*ast.FuncDecl) []string {
	var tableNames []string
	for _, fn := range funcDeclarations {
		ast.Inspect(fn.Body, func(n ast.Node) bool {
			blockStmt, ok := n.(*ast.BlockStmt)
			if !ok {
				return true
			}
			for _, stmt := range blockStmt.List {
				switch stmt := stmt.(type) {
				case *ast.ReturnStmt:
					tableName := nameFromReturnStmt(stmt)
					if tableName != "" {
						tableNames = append(tableNames, tableName)
					}
				case *ast.AssignStmt:
					// e.g. tableName := "<string>" or table := schema.Table{}
					tableName := nameFromAssignStmt(stmt)
					if tableName != "" {
						tableNames = append(tableNames, tableName)
					}
				case *ast.DeclStmt:
					// e.g. const tableName = "<string>"
					genDecl, ok := stmt.Decl.(*ast.GenDecl)
					if !ok || genDecl.Tok != token.CONST || len(genDecl.Specs) != 1 {
						return true
					}
					valueSpec, ok := genDecl.Specs[0].(*ast.ValueSpec)
					if !ok || len(valueSpec.Names) != 1 || !isTableName(valueSpec.Names[0].Name) || len(valueSpec.Values) != 1 {
						return true
					}
					basicLit, ok := valueSpec.Values[0].(*ast.BasicLit)
					if !ok {
						return true
					}
					tableName := trimTableName(basicLit.Value)
					tableNames = append(tableNames, tableName)
				}
			}
			return true
		})
	}
	return tableNames
}

var filesToSkip = map[string]bool{
	"resources/plugin/client.go":          true,
	"resources/plugin/plugin.go":          true,
	"resources/plugin/plugin_fips.go":     true,
	"resources/plugin/test_connection.go": true,
	"resources/plugin/tables.go":          true,
	"resources/plugin/cdc.go":             true,
}

func parseCodeTables() map[string]string {
	pathPrefix := "../../"
	tablesMap := make(map[string]string)
	goSourcePlugins, err := doublestar.Glob(os.DirFS(pathPrefix), "plugins/source/**/go.mod", doublestar.WithFailOnPatternNotExist(), doublestar.WithFilesOnly())
	if err != nil {
		log.Fatal(err)
	}
	for _, goSourcePlugin := range goSourcePlugins {
		if strings.HasPrefix(goSourcePlugin, "plugins/source/test") {
			continue
		}
		if strings.HasSuffix(goSourcePlugin, "_test.go") {
			continue
		}
		pluginDir := pathPrefix + strings.TrimSuffix(goSourcePlugin, "/go.mod")
		allGoFiles, err := doublestar.Glob(os.DirFS(pluginDir), "**/*.go", doublestar.WithFailOnPatternNotExist(), doublestar.WithFilesOnly())
		if err != nil {
			log.Fatal(err)
		}
		for _, goFile := range allGoFiles {
			if strings.HasSuffix(goFile, "_test.go") || !strings.HasPrefix(goFile, "resources") || filesToSkip[goFile] {
				continue
			}
			fset := token.NewFileSet()
			file, err := parser.ParseFile(fset, pluginDir+"/"+goFile, nil, parser.ParseComments)
			if err != nil {
				log.Fatal(err)
			}
			normalizedSourceFile := strings.TrimPrefix(pluginDir+"/"+goFile, pathPrefix)
			tablesFunctionsDeclarations := getTableFunctionDeclarations(file)
			if len(tablesFunctionsDeclarations) == 0 {
				continue
			}
			tableNames := findTableNamesInFunctionsDeclarations(tablesFunctionsDeclarations)
			tableNames = append(tableNames, findTableNamesInVariablesAndConstDeclarations(file)...)

			if len(tableNames) == 0 {
				// This error is not fatal because some tables have dynamic names like fmt.Sprintf("homebrew_analytics_build_errors_%s", days) but it's still useful to know for debugging
				fmt.Fprintf(os.Stderr, "Failed to find table name in file %s\n", normalizedSourceFile)
			}
			for _, tableName := range tableNames {
				tablesMap[tableName] = normalizedSourceFile
			}
		}

	}
	return tablesMap
}

func main() {
	tablesFromReadmes := parseDocsTables()
	tablesFromCode := parseCodeTables()

	for table, file := range tablesFromCode {
		if _, ok := tablesFromReadmes[table]; !ok {
			fmt.Printf("- Table `%s` is declared in code but missing from README. Table declaration file: `%s`\n", table, file)
		}
	}

	for table := range tablesFromReadmes {
		if _, ok := tablesFromCode[table]; !ok {
			// This error is not fatal because some tables have dynamic names like fmt.Sprintf("homebrew_analytics_build_errors_%s", days) but it's still useful to know for debugging
			fmt.Fprintf(os.Stderr, "- Table `%s` is in README but not found in code\n", table)
		}
	}
}
